/*
 * Intel QV Linux kernel driver
 * Copyright (c) 1999 - 2013, Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU General Public License,
 * version 2, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

/*
 *  Module Name:
 *    linuxdriverdevice_i.c
 *
 *  Abstract:
 *    This file contains implementation for any functions that are
 *    called from nalioctldrv.c and need to be run in kernel mode.
 *
 * VSS Revision Control Information:
 * ---------------------------------
 *   $Workfile: linuxdriverdevice_i.c $
 *   $Date: 2013/01/09 15:51:36 $
 *   $Archive: /QV2.0/nal/src/linux/driver/linuxdriverdevice_i.c $
 *   $Revision: 1.20 $
 */

#include <nalcodes.h>
#include <naltypes.h>
#include "nalioctl.h"
#include <osdevice_i.h>
#include <os_i.h>
#include "linuxnaldriver.h"
#include "linuxpci_i.h"
#include <linux/sched.h>       /* Interrupt routines */
#include <asm/io.h>             /* for I/O routines, memory mapping. */
#include <linux/pci.h>

#if !defined DMA_64BIT_MASK
#define DMA_64BIT_MASK  0xffffffffffffffffULL
#endif
#if !defined DMA_32BIT_MASK
#define DMA_32BIT_MASK  0x00000000ffffffffULL
#endif
/***************************************************************************
 * Local function prototypes.
 ***************************************************************************/
BOOLEAN
_NalCheckAndClearAdapterInterrupt(
    IN      NAL_MAC_TYPE    MacType,
    IN      KVOID*          BaseAddress
    );

void
_NalServiceRoutine(
    IN      int                 Irq,
    IN      void*               Context,
    IN      struct pt_regs*     Regs
    );

NAL_STATUS
_NalEnableDevice(
    IN      NAL_DEVICE_LOCATION     DeviceLocation,
    OUT     NAL_IO_RESOURCE*        IoResource,
    OUT     KVOID**                 Pdev
    );

NAL_STATUS
_NalFillDeviceResource(
    IN      NAL_DEVICE_LOCATION     DeviceLocation,
    OUT     NAL_IO_RESOURCE*        IoResource,
    OUT     KVOID**                 Pdev
    );

NAL_STATUS
_NalReleaseRegions(
    IN      KVOID*   PDev
    );
    
NAL_OS_RUN_DOMAIN 
_NalGetOsRunDomain(void);

/***************************************************************************
**
** Name:            _NalServiceRoutine()
**
** Description:     This is the interrupt service routine for NAL Windows. All it
**                  does is set the interrupt flag to TRUE so that NAL knows an
**                  interrupt has been triggered.
**
** Author:          MVM
**
** Born on Date:    03/24/2003
**
** Arguments:       Irq     -  The interrupt number.
**                  Context -  User specified object to pass to the handler.
**                             This contains our NAL_LINUX_SERVICE_ROUTINE_STRUCTURE
**                  Regs    -  Passed in by the OS. We don't use this.
**
** Returns:         Nothing.
**
****************************************************************************/
void
_NalServiceRoutine(
    IN      int                 Irq,
    IN      void*               Context,
    IN      struct pt_regs*     Regs
    )
{
    NAL_LINUX_ISR_DEVICE*  NalIsrDevice = (NAL_LINUX_ISR_DEVICE*)Context;
    /* Make sure the signature matches. Each shared interrupt is passed the context
     * value. Make sure this is ours */
    if(NalIsrDevice != NULL)
    {
        /* Determine if our device is interrupting and make it stop. The return value
         * to this is stored in the interrupt structure which can be tested to see if
         * the device did, indeed, interrupt. */
        NalIsrDevice->DeviceInterrupted =
            _NalCheckAndClearAdapterInterrupt((NAL_MAC_TYPE)NalIsrDevice->MacType,
                                              NalIsrDevice->HardwareVirtualAddress);
    }
}

/***************************************************************************
**
** Name:            _NalInitializeInterrupts()
**
** Description:     Tells NAL to be ready to perform the interrupt test.
**
** Author:          MVM
**
** Born on Date:    03/11/2003
**
** Arguments:       NalIsrDevice - Pointer to structure needed for the ISR Registration.
**
** Returns:         NAL_SUCCESS = ServiceRoutine registered at Vector successfully.
**                  NAL_VECTOR_INITIALIZATION_FAILED = ISR was unable to be registered.
**
****************************************************************************/
NAL_STATUS
_NalInitializeInterrupts(
    IN      NAL_LINUX_ISR_DEVICE*  NalIsrDevice
    )
{
#if 0
    NAL_STATUS  NalStatus  = NAL_INVALID_PARAMETER;
    int         Status     = 0;

    if(NalIsrDevice != NULL && NalIsrDevice->Signature == NAL_LINUX_INTERRUPT_SIGNATURE)
    {
        Status = request_irq(NalIsrDevice->Irq,
                             _NalServiceRoutine,         /* Pointer to the ISR     */
                             (SA_INTERRUPT | SA_SHIRQ),  /* Shared, fast interrupt */
                             NAL_LINUX_DRIVER_NAME,
                             (void*)NalIsrDevice);
        if(Status == 0)
        {
            NalStatus = NAL_SUCCESS;
        }
        else
        {
            NalStatus = NAL_VECTOR_INITIALIZATION_FAILED;
        }
    }

    return NalStatus;
#endif
    return NAL_NOT_IMPLEMENTED;
}

/***************************************************************************
**
** Name:            _NalUninitializeInterrupts()
**
** Description:     Informs the NAL that the interrupt test is over and any
**                  ISR registration can be released.
**
** Author:          MVM
**
** Born on Date:    03/11/2003
**
** Arguments:       NalIsrDevice - Pointer to structure needed for the ISR deregistration.
**
** Returns:         NAL_SUCCESS = Vector has been uninitialized successfully.
**                  NAL_INVALID_VECTOR = Vector was never initialized.
**
****************************************************************************/
NAL_STATUS
_NalUninitializeInterrupts(
    IN      NAL_LINUX_ISR_DEVICE*  NalIsrDevice
    )
{
#if 0
    NAL_STATUS  NalStatus  = NAL_INVALID_PARAMETER;

    if(NalIsrDevice != NULL && NalIsrDevice->Signature == NAL_LINUX_INTERRUPT_SIGNATURE)
    {
        free_irq(NalIsrDevice->Irq, NalIsrDevice);
        NalStatus = NAL_SUCCESS;
    }
    return NalStatus;
#endif
    return NAL_NOT_IMPLEMENTED;
}

/***************************************************************************
**
** Name:            _NalHasInterruptOccurred()
**
** Description:     Checks if an interrupt has occured for this adapter.
**
** Author:          MVM
**
** Born on Date:    03/11/2003
**
** Arguments:       NalAdapter - Pointer to the NAL_ADAPTER_STRUCTURE.
**
** Returns:         TRUE = Vector has received an interrupt and it has been cleared.
**                  FALSE = Vector has not received an interrupt.
**
****************************************************************************/
BOOLEAN
_NalHasInterruptOccurred(
    IN      NAL_LINUX_ISR_DEVICE*  NalIsrDevice
    )
{
    BOOLEAN InterruptOccurred = FALSE;

    if(NalIsrDevice != NULL && NalIsrDevice->Signature == NAL_LINUX_INTERRUPT_SIGNATURE)
    {
        InterruptOccurred = NalIsrDevice->DeviceInterrupted;
    }

    return InterruptOccurred;
}

/***************************************************************************
**
** Name:            _NalCheckAndClearAdapterInterrupt()
**
** Description:     This determines if the adapter is interrupting and if so, clears
**                  its interrupts.  Currently, this needs to be updated for each
**                  supported type of hardware. This needs to be fixed in the future
**                  as this file should not have to worry about specific hardware types.
**
** Author:          MVM
**
** Born on Date:    03/24/2003
**
** Arguments:       MacType     - The MAC Type.
**                  BaseAddress - Address where adapter is mapped.
**
** Returns:         TRUE if our device interrupted, FALSE otherwise.
**
****************************************************************************/
BOOLEAN
_NalCheckAndClearAdapterInterrupt(
    IN      NAL_MAC_TYPE    MacType,
    IN      KVOID*          BaseAddress
    )
{
#if 0
    UINT32      IcrRegister              = 0;
    UINT32      ImcRegister              = 0;
    UINT8       StatusRegisterHighByte   = 0;
    UINT8       CommandRegisterHighByte  = 0;
    BOOLEAN     OurInterrupt             = FALSE;

    /* Gig */
    if(MacType > NAL_MAC_TYPE_PRO1000_BEGIN &&
       MacType < NAL_MAC_TYPE_PRO1000_END)
    {
        /* We will read the IMC Register and store this value. Then we will clear all
         * interrupts through the IMC to prevent further interrupts from coming in
         * while we're here. If this routine is run because our adapter is interrupting,
         * we wont bother reenabling interrupts since the test is done. But if its not
         * our adapter, we'll enable again so that we can still interrupt since the
         * test is not yet done. */
        ImcRegister = readl((u32)((UINT8*)BaseAddress + E1000_IMC));
        if (MacType == NAL_MAC_TYPE_I82542_REV_2)
        {
            /******************************************************************
             * Disable all adapter interrupts (except Sequence errors since
             * that could mean that the cable has disconnected) by clearing all
             * bit in the interrupt mask.
             *****************************************************************/
            writel((0xFFFFFFFF & ~E1000_IMC_RXSEQ), ((u32)((UINT8*)BaseAddress) + E1000_IMC));
        }
        else
        {
            /* Disable all adapter interrupts by clearing all bit in the
             * interrupt mask
             */
            writel(NAL_8254X_IRQ_CLEAR_MASK, ((u32)((UINT8*)BaseAddress) + E1000_IMC));
        }

        IcrRegister = readl((u32)((UINT8*)BaseAddress + E1000_ICR));
        if(IcrRegister != 0)
        {
            OurInterrupt = TRUE;
        }
        else
        {
            /* This was not our interrupt so we will reenable interrupts in the IMC */
            writel(ImcRegister, ((u32)((UINT8*)BaseAddress) + E1000_IMC));
        }
    }

    /* 10 Gig */
    if(MacType > NAL_MAC_TYPE_PRO10GBE_BEGIN &&
       MacType < NAL_MAC_TYPE_PRO10GBE_END)
    {
        IcrRegister = readl((u32)((UINT8*)BaseAddress + IXGB_ICR));
        if(IcrRegister != 0)
        {
            OurInterrupt = TRUE;
        }
    }

    /* 10/100 */
    if(MacType > NAL_MAC_TYPE_PRO100_BEGIN &&
       MacType < NAL_MAC_TYPE_PRO100_END)
    {

        /* Check if this is our interrupt */
        StatusRegisterHighByte = readb((u32)((UINT8*)BaseAddress + CSR_SCB_STATUS_HIGH));
        if(StatusRegisterHighByte)
        {
            OurInterrupt = TRUE;

            /* Mask off all interrupts */
            CommandRegisterHighByte = readb((u32)((UINT8*)BaseAddress + CSR_SCB_COMMAND_HIGH));
            CommandRegisterHighByte |= BIT_0;
            writeb(CommandRegisterHighByte, ((u32)((UINT8*)BaseAddress + CSR_SCB_COMMAND_HIGH)));

            /* Clear the interrupt */
            writeb(StatusRegisterHighByte, ((u32)((UINT8*)BaseAddress + CSR_SCB_STATUS_HIGH)));
        }
        /* Sometimes it is possible that a Int generated but CSR_SCB_STATUS_HIGH register still waiting for update.
           This is spurious interrupt */
        else
        {
            StatusRegisterHighByte = readb((u32)((UINT8*)BaseAddress + CSR_SCB_STATUS_HIGH));
            if(StatusRegisterHighByte)
            {
                OurInterrupt = TRUE;

                /* Mask off all interrupts */
                CommandRegisterHighByte = readb((u32)((UINT8*)BaseAddress + CSR_SCB_COMMAND_HIGH));
                CommandRegisterHighByte |= BIT_0;
                writeb(CommandRegisterHighByte, ((u32)((UINT8*)BaseAddress + CSR_SCB_COMMAND_HIGH)));

                /* Clear the interrupt */
                writeb(StatusRegisterHighByte, ((u32)((UINT8*)BaseAddress + CSR_SCB_STATUS_HIGH)));
            }
        }
    }

    return OurInterrupt;
#endif
    return FALSE;
}

/***************************************************************************
**
** Name:            NalResolveOsSpecificIoctl()
**
** Description:     This handles IOCTL for Linux specific functions that must
**                  be performed in Kernel mode and called from within user mode.
**
** Author:          MVM
**
** Born on Date:    01/23/2002
**
** Arguments:       NalIoctl - Pointer to a NAL_IOCTL_INPUT_DATA structure containing all
**                             relevent data for the IOCTL.
**
** Returns:         NAL Status Code. This is the status code for
**                  NalResolveOsSpecificIoctl, not for the individual functions called
**                  from within.
**
****************************************************************************/
NAL_STATUS
NalResolveOsSpecificIoctl(
    IN      NAL_IOCTL_INPUT_DATA*  NalIoctl
    )
{
    NAL_STATUS NalStatus = NAL_INVALID_PARAMETER;

    if(NalIoctl != NULL)
    {
        switch(NalIoctl->FunctionId)
        {
           case NAL_LINUX_ADAPTER_IN_USE_FUNCID:
                {
                    NAL_LINUX_ADAPTER_IN_USE_FUNC* FunctionData =
                        (NAL_LINUX_ADAPTER_IN_USE_FUNC*)(&(NalIoctl->InputBuffer));

                    printk(KERN_DEBUG "calling ioctl In mark adapter lock \n");
                    if(FunctionData != NULL)
                    {
                        FunctionData->CanBeUsed = _NalMarkAdapterInUse(FunctionData->NalDevice,
                                                                       FunctionData->Locked);
                        NalStatus = NAL_SUCCESS;
                    }
                    else
                    {
                        NalStatus = NAL_INVALID_PARAMETER;
                        NalDebugPrint("NalResolveOsSpecificIoctl: NAL_LINUX_ADAPTER_IN_USE_FUNCID FunctionData is NULL\n");
                    }
                }
                break;
           case NAL_LINUX_IS_ADAPTER_IN_USE_FUNCID:
                {
                    NAL_LINUX_IS_ADAPTER_IN_USE_FUNC* FunctionData =
                        (NAL_LINUX_IS_ADAPTER_IN_USE_FUNC*)(&(NalIoctl->InputBuffer));

                    if(FunctionData != NULL)
                    {
                        FunctionData->IsInUse = _NalIsAdapterInUse(FunctionData->NalDevice);
                        NalStatus = NAL_SUCCESS;
                    }
                    else
                    {
                        NalStatus = NAL_INVALID_PARAMETER;
                        NalDebugPrint("NalResolveOsSpecificIoctl: NAL_LINUX_IS_ADAPTER_IN_USE_FUNCID FunctionData is NULL\n");
                    }
                }
                break;
            case NAL_LINUX_REQUEST_REGIONS_FUNCID:
                {
                    NAL_LINUX_DEVICERESOURCE_FUNC* FunctionData =
                        (NAL_LINUX_DEVICERESOURCE_FUNC*)(&(NalIoctl->InputBuffer));

                    if(FunctionData != NULL)
                    {
                        FunctionData->ReturnValue = _NalEnableDevice(FunctionData->DeviceLocation,
                                                                     &(FunctionData->NalIoResource[0]),
                                                                     &FunctionData->Pdev);
                        NalStatus = NAL_SUCCESS;
                    }
                    else
                    {
                        NalStatus = NAL_INVALID_PARAMETER;
                        NalDebugPrint("NalResolveOsSpecificIoctl: NAL_LINUX_REQUEST_REGIONS_FUNCID FunctionData is NULL\n");
                    }
                }
                break;
            case NAL_LINUX_RELEASE_REGIONS_FUNCID:
                {
                    NAL_LINUX_DEVICERESOURCE_FUNC* FunctionData =
                        (NAL_LINUX_DEVICERESOURCE_FUNC*)(&(NalIoctl->InputBuffer));
                    NalDebugPrint("NalResolveOsSpecificIoctl: NAL_LINUX_RELEASE_REGIONS_FUNCID FunctionData is NULL\n");

                    if(FunctionData != NULL)
                    {
                        FunctionData->ReturnValue = _NalReleaseRegions(FunctionData->Pdev);
                        NalStatus = NAL_SUCCESS;
                    }
                    else
                    {
                        NalStatus = NAL_INVALID_PARAMETER;
                        NalDebugPrint("NalResolveOsSpecificIoctl: NAL_LINUX_RELEASE_REGIONS_FUNCID FunctionData is NULL\n");
                    }
                }
                break;

            case NAL_LINUX_INC_DRIVER_REFCOUNT_FUNCID:
                {
                    _NalDriverIncrementReferenceCount();
                    NalStatus = NAL_SUCCESS;
                }
                break;

            case NAL_LINUX_DEC_DRIVER_REFCOUNT_FUNCID:
                {
                    _NalDriverDecrementReferenceCount();
                    NalStatus = NAL_SUCCESS;
                }
                break;

            case NAL_LINUX_GET_DRIVER_REFCOUNT_FUNCID:
                {
                    NAL_LINUX_REFCOUNT_FUNCS* FunctionData =
                        (NAL_LINUX_REFCOUNT_FUNCS*)(&(NalIoctl->InputBuffer));

                    if(FunctionData != NULL)
                    {
                        FunctionData->ReturnValue =  _NalDriverGetReferenceCount();
                        NalStatus = NAL_SUCCESS;
                    }
                    else
                    {
                        NalStatus = NAL_INVALID_PARAMETER;
                    }
                }
                break;

            case NAL_LINUX_DRIVER_GET_VERSION:
                {
                    NAL_LINUX_DRIVER_GET_VERSION_FUNCS* FunctionData =
                        (NAL_LINUX_DRIVER_GET_VERSION_FUNCS*)(&(NalIoctl->InputBuffer));

                   NalDebugPrint("NalResolveOsSpecificIoctl: Nal get version Ioctl\n");

                    if(FunctionData != NULL)
                    {
                        _NalDriverGetVersion(FunctionData->Version);
                        NalStatus = NAL_SUCCESS;
                    }
                    else
                    {
                        NalStatus = NAL_INVALID_PARAMETER;
                    }
                }
                break;

            case NAL_LINUX_FILL_DEVICE_RESOURCE_FUNCID:
                {
                    NAL_LINUX_DEVICERESOURCE_FUNC* FunctionData =
                        (NAL_LINUX_DEVICERESOURCE_FUNC*)(&(NalIoctl->InputBuffer));

                    NalDebugPrint("NalResolveOsSpecificIoctl: Nal Fill device resource Ioctl\n");

                    if(FunctionData != NULL)
                    {
                        FunctionData->ReturnValue = _NalFillDeviceResource(FunctionData->DeviceLocation, 
                                                                           &(FunctionData->NalIoResource[0]),
                                                                           &FunctionData->Pdev);
                        NalStatus = NAL_SUCCESS;
                    }
                    else
                    {
                        NalStatus = NAL_INVALID_PARAMETER;
                        NalDebugPrint("NalResolveOsSpecificIoctl: NAL_LINUX_FILL_DEVICE_RESOURCE_FUNCID FunctionData is NULL\n");
                    }
                }
                break;

            case NAL_LINUX_FREEMEMORYNONPAGEDPCI_FUNCID:
                {
                    NAL_LINUX_FREEMEMORYNONPAGEDPCI_FUNC* FunctionData =
                        (NAL_LINUX_FREEMEMORYNONPAGEDPCI_FUNC*)(&(NalIoctl->InputBuffer));

                        NalFreeMemoryNonPagedPci(FunctionData->PDev,
                                               FunctionData->Address,
                                               FunctionData->PhysicalAddress,
                                               FunctionData->Size);

                    NalStatus = NAL_SUCCESS;
                }
                break;

            case NAL_LINUX_ALLOCATEMEMORYNONPAGEDPCI_FUNCID:
                {
                    NAL_LINUX_ALLOCATEMEMORYNONPAGEDPCI_FUNC* FunctionData =
                        (NAL_LINUX_ALLOCATEMEMORYNONPAGEDPCI_FUNC*)(&(NalIoctl->InputBuffer));

                        FunctionData->ReturnValue =
                            _NalAllocateMemoryNonPagedPci(FunctionData->PDev,
                                                       FunctionData->ByteCount,
                                                       FunctionData->Alignment,
                                                       &FunctionData->PhysicalAddress,
                                                       NULL,
                                                       0);

                    NalStatus = NAL_SUCCESS;
                }
                break;

            case NAL_LINUX_READPCIEXBYTE_FUNCID:
                {
                    NAL_LINUX_READPCIEXBYTE_FUNC* FunctionData =
                        (NAL_LINUX_READPCIEXBYTE_FUNC*)(&(NalIoctl->InputBuffer));

                    FunctionData->ReturnValue =
                        _NalOsReadPciExByte(FunctionData->PciLocation,
                                                   FunctionData->ByteIndex,
                                                   &FunctionData->Value);


                    NalStatus = NAL_SUCCESS;
                }
                break;

            case NAL_LINUX_WRITEPCIEXBYTE_FUNCID:
                {
                    NAL_LINUX_WRITEPCIEXBYTE_FUNC* FunctionData =
                        (NAL_LINUX_WRITEPCIEXBYTE_FUNC*)(&(NalIoctl->InputBuffer));
                    FunctionData->ReturnValue =
                        _NalOsWritePciExByte(FunctionData->PciLocation,
                                                    FunctionData->ByteIndex,
                                                    FunctionData->Value);

                    NalStatus = NAL_SUCCESS;
                }
                break;
            case NAL_LINUX_GET_RUN_DOMAIN_FUNCID:
                {
                    NAL_LINUX_RUN_DOMAIN_FUNC* FunctionData =
                        (NAL_LINUX_RUN_DOMAIN_FUNC*)(&(NalIoctl->InputBuffer));

                    printk(KERN_DEBUG "calling ioctl get run domain\n");
                    if(FunctionData != NULL)
                    {
                        FunctionData->RunDomain = _NalGetOsRunDomain();
                        FunctionData->ReturnValue = NAL_SUCCESS;
                        NalStatus = NAL_SUCCESS;
                    }
                    else
                    {
                        NalStatus = NAL_INVALID_PARAMETER;
                        NalDebugPrint("NalResolveOsSpecificIoctl: NAL_LINUX_GET_RUN_DOMAIN_FUNCID FunctionData is NULL\n");
                    }
                }
                break;

           default:
                NalStatus = NAL_IOCTL_INVALID_FUNCTION_ID;
                break;
        }
    }

    return NalStatus;
}

/***************************************************************************
**
** Name:            _NalEnableDevice()
**
** Description:     This does everything that is required to activate the device.
*                   This requests the I/O and Memory regions for the device.
**                  This performs what is required for I/O to be performed on Linux.
**
** Author:          LKU
**
** Born on Date:    03/21/2011
**
** Arguments:       DeviceLocation - Pci location information for this device.
**
** Rerutns:         IoResource - List of resources.
**                  PDev - device handle in kernel.
**
**                  NAL Status Code. This is the status code for
**                  NalResolveOsSpecificIoctl, not for the individual functions called
**                  from within.
**
****************************************************************************/
NAL_STATUS
_NalEnableDevice(
    IN      NAL_DEVICE_LOCATION     DeviceLocation,
    OUT     NAL_IO_RESOURCE*        IoResource,
    OUT     KVOID**                 PDev
    )
{
    struct pci_dev* PciDevice       = NULL;
    struct pci_bus* LinuxPciBus     = NULL;
    NAL_STATUS      NalStatus       = NAL_SUCCESS;
    UINT32          i               = 0;
    unsigned int    DeviceFunction  = 0;
    int             Result          = 0;

    /* Set up the DeviceFunction variable */
    DeviceFunction = PCI_DEVFN(DeviceLocation.Pci.Device, DeviceLocation.Pci.Function);

    do
    {
        /* Use segments for creating a Linux PCi Device, for segemented architecture */
        LinuxPciBus = pci_find_bus(DeviceLocation.Pci.Segment, DeviceLocation.Pci.Bus);
        if(LinuxPciBus == NULL)
        {
            NalStatus = NAL_PCICONFIG_NOT_AVAILABLE;
            break;
        }

        /* Get the pci device from the bus/dev/function */
        PciDevice = pci_get_slot(LinuxPciBus, DeviceFunction);
        if(PciDevice == NULL)
        {
            NalStatus = NAL_PCICONFIG_NOT_AVAILABLE;
            break;
        }

        /* First enable the device. This wakes it up and enables memory and I/O */
        Result = pci_enable_device(PciDevice);
        if(Result != 0)
        {
            NalStatus = NAL_PCICONFIG_NOT_AVAILABLE;
            break;
        }

        /* Set a DMA Mask */
        if (pci_set_dma_mask(PciDevice, DMA_64BIT_MASK) ||
            pci_set_consistent_dma_mask(PciDevice, DMA_64BIT_MASK))
        {
            if (pci_set_dma_mask(PciDevice, DMA_32BIT_MASK) ||
                pci_set_consistent_dma_mask(PciDevice, DMA_32BIT_MASK))
            {
                    printk(KERN_DEBUG "No usable DMA configuration, aborting\n");
                    NalStatus = NAL_DMA_NOT_SUPPORTED;
                    break;
            }
        }

        for(i=0; i<6; i++)
        {
            /* Locate a valid BAR */
            if(pci_resource_len(PciDevice, i) == 0)
            {
                continue;
            }

            /* Request I/O or Mem resource depending on what type of resource it is
             * if it is not used already by the kernel. If it is already used
             * probably these I/O addresses have been used by other device.
             * Device resources conflict is unlikely to happen on modern PnP devices
             * but requesting the same region twice corrput the kernel internals anyway.*/
            if(pci_resource_flags(PciDevice, i) & IORESOURCE_IO)
            {
                request_region(pci_resource_start(PciDevice, i),
                               pci_resource_len(PciDevice, i),
                               NAL_LINUX_DRIVER_NAME);
                IoResource[i].MemoryAddress = pci_resource_start(PciDevice, i);
                IoResource[i].Type = NAL_IO_TYPE_IO;
            }
            else if(pci_resource_flags(PciDevice, i) & IORESOURCE_MEM)
            {
                if(!check_mem_region(pci_resource_start(PciDevice, i), pci_resource_len(PciDevice, i)))
                {
                    request_mem_region(pci_resource_start(PciDevice, i),
                                       pci_resource_len(PciDevice, i),
                                       NAL_LINUX_DRIVER_NAME);
                    IoResource[i].MemoryAddress = pci_resource_start(PciDevice, i);
                }
                else
                {
                    IoResource[i].MemoryAddress = 0;
                }
                IoResource[i].Type = NAL_IO_TYPE_MEM;
            }
        }

        /* Turn on bus mastering for the device first */
        pci_set_master(PciDevice);

        /* Now go ahead and store the pdev pointer */
        *PDev = PciDevice;
    } while(0);
    
    return NalStatus;
}

/***************************************************************************
**
** Name:            _NalFillDeviceResource()
**
** Description:     This fills the I/O and Memory regions for the device.
**                  This provides what is required for I/O to be performed on Linux.
**
** Author:          LKU
**
** Born on Date:    03/21/2011
**
** Arguments:       DeviceLocation - Pci location information for this device.
**
** Rerutns:         IoResource - List of resources.
**                  PDev - device handle in kernel.
**
**                  NAL Status Code. This is the status code for
**                  NalResolveOsSpecificIoctl, not for the individual functions called
**                  from within.
**
****************************************************************************/
NAL_STATUS
_NalFillDeviceResource(
    IN      NAL_DEVICE_LOCATION     DeviceLocation,
    OUT     NAL_IO_RESOURCE*        IoResource,
    OUT     KVOID**                 PDev
    )
{
    struct pci_dev* PciDevice       = NULL;
    struct pci_bus* LinuxPciBus     = NULL;
    NAL_STATUS      NalStatus       = NAL_SUCCESS;
    UINT32          i               = 0;
    unsigned int    DeviceFunction  = 0;

    printk(KERN_DEBUG "In Nal Fill Device resource\n");
    /* Set up the DeviceFunction variable */
    DeviceFunction = PCI_DEVFN(DeviceLocation.Pci.Device, DeviceLocation.Pci.Function);

    do
    {
        /* Use segments for creating a Linux PCi Device, for segemented architecture */
        LinuxPciBus = pci_find_bus(DeviceLocation.Pci.Segment, DeviceLocation.Pci.Bus);
        if(LinuxPciBus == NULL)
        {
            NalStatus = NAL_PCICONFIG_NOT_AVAILABLE;
            break;
        }

        /* Get the pci device from the bus/dev/function */
        PciDevice = pci_get_slot(LinuxPciBus, DeviceFunction);
        if(PciDevice == NULL)
        {
            NalStatus = NAL_PCICONFIG_NOT_AVAILABLE;
            break;
        }

        /* Check if a DMA is supported */
        if (!pci_dma_supported(PciDevice, DMA_64BIT_MASK))
        {
            if (!pci_dma_supported(PciDevice, DMA_32BIT_MASK))
            {
                printk(KERN_DEBUG "No usable DMA configuration, aborting\n");
                NalStatus = NAL_DMA_NOT_SUPPORTED;
                break;
            }
        }

        for(i=0; i<6; i++)
        {
            /* Locate a valid BAR */
            if(pci_resource_len(PciDevice, i) == 0)
            {
                continue;
            }

            /* Fill I/O or Mem resource depending on what type of resource it is
             * if it is already claimed in the kernel.*/
            if(pci_resource_flags(PciDevice, i) & IORESOURCE_IO)
            {
                IoResource[i].MemoryAddress = pci_resource_start(PciDevice, i);
                IoResource[i].Type = NAL_IO_TYPE_IO;
            }
            else if(pci_resource_flags(PciDevice, i) & IORESOURCE_MEM)
            {
                if(!check_mem_region(pci_resource_start(PciDevice, i), pci_resource_len(PciDevice, i)))
                {
                    IoResource[i].MemoryAddress = 0;
                }
                else
                {
                    IoResource[i].MemoryAddress = pci_resource_start(PciDevice, i);
                }
                IoResource[i].Type = NAL_IO_TYPE_MEM;
            }
        }

        /* Now go ahead and store the pdev pointer */
        *PDev = PciDevice;
    }while(0);

    return NalStatus;
}

/***************************************************************************
**
** Name:            _NalReleaseRegions()
**
** Description:     This releases the I/O and Memory regions for the device.
**
** Author:          LKU
**
** Born on Date:    03/21/2011
**
** Arguments:       PDev - device handle in kernel.
**
** Returns:         NAL Status Code. This is the status code for
**                  NalResolveOsSpecificIoctl, not for the individual functions called
**                  from within.
**
****************************************************************************/
NAL_STATUS
_NalReleaseRegions(
    IN      KVOID*   PDev
    )
{
    struct pci_dev* PciDevice       = (struct pci_dev*)PDev;
    NAL_STATUS      NalStatus       = NAL_SUCCESS;
    UINT32          i               = 0;

    for(i=0; i<6; i++)
    {
        /* Locate a valid BAR */
        if(pci_resource_len(PciDevice, i) == 0)
        {
            continue;
        }

        /* Release I/O or Mem resource depending on what type of resource it is */
        if(pci_resource_flags(PciDevice, i) & IORESOURCE_IO)
        {
            release_region(pci_resource_start(PciDevice, i),
                           pci_resource_len(PciDevice, i));
        }
        else if(pci_resource_flags(PciDevice, i) & IORESOURCE_MEM)
        {
            release_mem_region(pci_resource_start(PciDevice, i),
                               pci_resource_len(PciDevice, i));
        }
    }
    pci_dev_put(PciDevice);
    pci_disable_device(PciDevice);

    return NalStatus;
}

/***************************************************************************
**
** Name:            _NalGetOsRunDomain()
**
** Description:     This returns in which priviledge domain kernel runs.
**
** Author:          LKU
**
** Born on Date:    03/24/2011
**
** Arguments:       NONE
**
** Returns:         NAL_OS_RUN_DOMAIN. This is one of the priviledge domains.
**                  
**
****************************************************************************/
NAL_OS_RUN_DOMAIN 
_NalGetOsRunDomain()
{
    NAL_OS_RUN_DOMAIN    Domain = NAL_OS_DOMAIN_BAREMETAL;

#ifdef is_running_on_xen
    if(is_running_on_xen())
    {
        if(is_initial_xendomain())
        {
            Domain = NAL_OS_DOMAIN_0;
        }
        else
        {
            Domain = NAL_OS_DOMAIN_U;
        }
    }
#endif

    return Domain;
}

