#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <signal.h>
#include <fcntl.h>
#include <ctype.h>
#include <asm/types.h>
#include <sys/ioctl.h>
#include <sys/wait.h>
#include <sys/mman.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netinet/ether.h>
#include <net/if.h>
#include <arpa/inet.h>
#include <net/route.h>
#include <asm/types.h>
#include <linux/netlink.h>
#include <linux/rtnetlink.h>
//#include "iscsi.h"
#include "iscsi-ioctl.h"
#include "fwparam_ibft.h"

#ifndef TRUE
#define TRUE 0
#endif

#ifndef FALSE
#define FALSE 1
#endif

#ifndef BOOL
#define BOOL int
#endif

#define ISCSIDEV "/dev/iscsictl"
char *progname = "iscsi_boot_util";
static int iscsifd = -1;
int debug = -1;

typedef struct
{
	char ifname[IFNAMSIZ]; 

	BOOL enabled;
	BOOL primary;

	unsigned char *n_ip;
	unsigned char *n_mac;  
	unsigned long n_net;
	unsigned char *n_gate;

} netdev_t;

#define OFFSET_TO_STRING(off) ((char*)off)

#define OPTION_ROM_SIGNATURE         0xAA55
#define START_FROM                   0xC0000
#define SEARCH_TILL                  0xF0000
#define MAP_SIZE                     (SEARCH_TILL - START_FROM + 1)

// Speced length is 223 + 1 for the NULL char
#define MAX_IQN_LEN                  224
#define NUM_ETHERNET_INTERFACES       32
#define MAX_NETDEV                    20
#define MAC_LENGTH                     6
#define MAX_NETMASK_LENGTH            16

#define ETHSETTINGS_FILE             "/etc/ethsettings"

#define ETH_MASK                     0x01
#define DISCOVERY_MASK               0x02
#define IF_MASK                      0x04
#define SILENT_MASK                  0x08

#define PRINT_DBG_MSG do {							\
    fprintf(stderr, "iscsi_util:iscsi boot failure at line %u, file %s (%u) [%s]\n", \
	    __LINE__, __FILE__,						\
	    errno, strerror(errno)); exit(1); } while(0)

#define iscsi_printf(fmt, args...) if (SILENT_MASK != (flags & SILENT_MASK)) { printf("iSCSI Boot Utility: "fmt, ##args); }

BOOL ipToAddress(const u_int32_t* ip, char* address);
BOOL prefixToNetmask(u_int32_t prefix, char* netmask);
BOOL writeNetworkInformation(const netdev_t* nic);
char *determineTableOffset(const unsigned long  map_base);
int determineOffsets( void *ibft_loc);
BOOL ifup(const netdev_t* nic);
int iscsiIoctl ( int request, void *data ); 
void iscsiSetDefaults ( struct iscsi_session_ioctl *session ); 
int connectToTarget ( void* offset ); 

u_int8_t flags       = 0;
static int sockfd = -1;
netdev_t *nicdev0;
netdev_t *nicdev1;

int main(int argc, char **argv) 
{
				int      fd          = 0;
				unsigned long      map_base    = 0;
				char* offset;
				int      option      = 0;

				while ((option = getopt (argc, argv, "ecduasv")) != -1)
				{
								switch (option)
								{  
												case 'e':
																flags |= ETH_MASK;
																break;

												case 'u':
																flags |= IF_MASK;
																break; 

												case 's':
																flags |= SILENT_MASK;
																break;

												case 'd':
																flags |= DISCOVERY_MASK;
																break;

												case 'v':
																debug++;
																break;

												case 'a':
												default:
																flags |= ETH_MASK;
																flags |= IF_MASK;
																flags |= DISCOVERY_MASK;
								}
				}


				// Open the physical memory device file
				if((fd = open("/dev/mem", O_RDONLY )) == -1)
				{
								iscsi_printf("Failed to open /dev/mem\n");
								return FALSE;
				}

				// virtual Mapping of RAM
				map_base = (unsigned long)mmap(0, MAP_SIZE , PROT_READ , MAP_SHARED, fd, START_FROM);

				if (MAP_FAILED == (void*)map_base)
				{
								iscsi_printf("mmap failed. Errno: %u\n", errno);
								munmap((void*)map_base, MAP_SIZE); 
								close(fd);
								return FALSE;
				}

				if (!(offset= determineTableOffset(map_base)))
				{
								iscsi_printf("Unable to determine the table offset\n");
								close(fd);
								return FALSE;
				}

				if (determineOffsets(offset))
				{
								iscsi_printf("iBFT is not configured correctly\n");
				}

				//populating the netdev_t structures


								iscsi_printf("PRASANNAPRASANNA\n");

				// It's horrible, but this application will continue to write duplicate entries
				// into the ethsettings file. I'm just going to > /etc/ethsettings before running
				// this application, but it really should be fixed here.
				if (ETH_MASK == (flags & ETH_MASK))
				{
								if (TRUE == nicdev0->enabled)
								{
												if (TRUE == nicdev0->primary)
												{
																writeNetworkInformation(nicdev0); 

																if (TRUE == nicdev1->enabled)
																{
																				writeNetworkInformation(nicdev1);
																}
												}
								}

								if (TRUE == nicdev1->enabled)
								{
												if (TRUE == nicdev1->primary)
												{
																writeNetworkInformation(nicdev1);

																if (TRUE == nicdev0->enabled)
																{
																				writeNetworkInformation(nicdev0);
																}
												}
								}
				}

				if (IF_MASK == (flags & IF_MASK))
				{
								if (TRUE == nicdev0->enabled)
								{
												ifup(nicdev0);
								}

								if (TRUE == nicdev1->enabled)
								{
												ifup(nicdev1);
								}
				}

				// Do discovery
				if ( flags & DISCOVERY_MASK )
				{
								iscsi_printf("\nStarting Target Discovery...");
								connectToTarget( offset);
				}

				// cleanup 
				munmap((void*)map_base, MAP_SIZE); 
				close(fd);
				return TRUE;
}

void
verify_hdr(char *name, struct ibft_hdr *hdr, int id, int version, int length)
{
#define VERIFY_HDR_FIELD(val) \
        if (hdr->val != val) { \
                fprintf(stderr, \
                        "%s: error, %s structure expected %s %d but" \
                        " got %d\n", \
                        progname, name, #val, hdr->val, val); \
                exit(1); \
        }

        if (debug > 1)
                fprintf(stderr, "%s: verifying %s header\n", __FUNCTION__,
                        name);

        VERIFY_HDR_FIELD(id);
        VERIFY_HDR_FIELD(version);
        VERIFY_HDR_FIELD(length);

#undef VERIFY_HDR_FIELD
}

#define CHECK_HDR(ident, name) \
        verify_hdr(#name, &ident->hdr, id_##name, version_##name, \
                   sizeof(*ident))


BOOL writeNetworkInformation(const netdev_t* nic)
{
				FILE* fd_ethsettings;
				int index;
				int sock;
				char iscsi_mac[200];
				char iface_mac[200];
				char ethX[IFNAMSIZ];
				struct ether_addr iscsi_eth;
				struct ether_addr eth;
				struct ifreq ifr;

				memset(&iscsi_eth, 0, sizeof(struct ether_addr));
				memcpy(iscsi_eth.ether_addr_octet, (char*)(nic->n_mac), MAC_LENGTH);

				sock = socket(PF_INET,SOCK_DGRAM, 0);

				if (-1 == sock)
				{
								iscsi_printf("Could not create a socket\n");
				}

				for(index = 0 ; index < NUM_ETHERNET_INTERFACES ; index++) 
				{
								memset(ethX, 0, IFNAMSIZ);
								memset(&ifr, 0, sizeof(struct ifreq));
								sprintf(ethX, "eth%d", index);
								strcpy(ifr.ifr_name, ethX);      

								if(0 > ioctl(sock, SIOCGIFHWADDR, &ifr))
								{
												iscsi_printf("Could not get network device properties\n");
								}

								memset(&eth, 0, sizeof(struct ether_addr));
								memcpy(eth.ether_addr_octet, (char*)(ifr.ifr_hwaddr.sa_data), MAC_LENGTH);

								strcpy(iscsi_mac, ether_ntoa(&iscsi_eth));
								strcpy(iface_mac, ether_ntoa(&eth));

								if (0 == strcmp(iscsi_mac, iface_mac))
								{
												strcpy((char*)nic->ifname, ifr.ifr_name);

												if ((fd_ethsettings = fopen(ETHSETTINGS_FILE, "a")) != NULL)
												{
																fprintf(fd_ethsettings, "%s\n", ethX);
																fflush(fd_ethsettings);
																fclose(fd_ethsettings);
												}
												else
												{
																iscsi_printf("Failed to open %s\n", ETHSETTINGS_FILE);
												}
												break;
								}
				}

				return TRUE;
}

void
format_ipaddr(char *buf, size_t size, uint8_t *ip)
{
        if (ip[0] == 0 && ip[1] == 0 && ip[2] == 0 && ip[3] == 0 &&
            ip[4] == 0 && ip[5] == 0 && ip[6] == 0 && ip[7] == 0 &&
            ip[8] == 0 && ip[9] == 0 && ip[10] == 0xff && ip[11] == 0xff) {
                /*
                 * IPV4
                 */
                snprintf(buf, size, "%d.%d.%d.%d", ip[12], ip[13], ip[14], ip[15]);
        } else {
                /* XXX ... */
                fprintf(stderr, "%s: warning no IPV6 support.\n", progname);
                buf[0] = '\0';
                return;
        }

}


int connectToTarget ( void* ibft_hdr ) 
{
	struct iscsi_session_ioctl session;
	char  *target_ip=NULL; 
	struct sockaddr_in *sin =
		( struct sockaddr_in * ) &session.portal.addr;
	char ipbuf[32];
	struct ibft_table_hdr *ibft_loc = ibft_hdr;
	struct ibft_control *control;
	struct ibft_initiator *initiator = NULL;
	struct ibft_tgt *tgt0 = NULL, *tgt1 = NULL;


	memset ( &session, 0, sizeof ( session ) );

	control = ibft_hdr + sizeof(*ibft_loc);
	CHECK_HDR(control, control);

	if (control->initiator_off) 
		initiator = ibft_hdr + control->initiator_off;

	if (control->tgt0_off) 
		tgt0 = ibft_hdr + control->tgt0_off;

	if (control->tgt1_off) 
		tgt1 = ibft_hdr + control->tgt1_off;

	iscsiSetDefaults ( &session );
	session.ioctl_version = ISCSI_SESSION_IOCTL_VERSION;

	strncpy((char *)session.initiator_name,
			(char *)ibft_hdr+initiator->initiator_name_off,
			initiator->initiator_name_len + 1);

	strncpy((char *)session.initiator_alias,
			(char *)ibft_hdr+initiator->initiator_name_off,
			initiator->initiator_name_len + 1);

	if (tgt0 && (tgt0->hdr.flags & INIT_FLAG_FW_SEL_BOOT)) {
		strncpy((char *)session.target_name,
				(char *)(ibft_hdr+tgt0->tgt_name_off),
				tgt0->tgt_name_len);
		format_ipaddr(ipbuf, sizeof(ipbuf), tgt0->ip_addr);
		target_ip =ipbuf;
		sin->sin_port = htons(tgt0->port);
		session.password_length=tgt0->chap_name_len;
		strncpy(session.username,
				(char *)(ibft_hdr + tgt0->chap_name_off),
				tgt0->chap_name_len);
		strncpy((char *)session.password,
				(char*)(ibft_hdr + tgt0->chap_secret_off),
				tgt0->chap_secret_len);
	} else if (tgt1 && (tgt1->hdr.flags & INIT_FLAG_FW_SEL_BOOT)) {
		strncpy((char *)session.target_name,
				(char *)(ibft_hdr+tgt1->tgt_name_off),
				tgt1->tgt_name_len);
		format_ipaddr(ipbuf, sizeof(ipbuf), tgt1->ip_addr); 
		target_ip =ipbuf;
		sin->sin_port = htons(tgt1->port);
		session.password_length=tgt1->chap_name_len;
		strncpy(session.username,
				(char *)(ibft_hdr + tgt1->chap_name_off),
				tgt1->chap_name_len);
		strncpy((char *)session.password,
				(char*)(ibft_hdr + tgt1->chap_secret_off),
				tgt1->chap_secret_len);
	}
	sin->sin_family = AF_INET;

	//Setting up default values for max_recv_data_segment_len, first_burst_len
	//and max_burst_len

	session.portal.max_recv_data_segment_len=8192;
	session.portal.first_burst_len=65536;
	session.portal.max_burst_len=262144;

	inet_aton(target_ip, &(sin->sin_addr));

	return iscsiIoctl ( ISCSI_ESTABLISH_SESSION, &session );
}

void iscsiSetDefaults ( struct iscsi_session_ioctl *session ) 
{
	static const char isid[6] = { 0x40, 0x00, 0x27, 0x23, 0x00, 0x00 };

	memset ( session, 0, sizeof ( *session ) );

	session->ioctl_version = ISCSI_SESSION_IOCTL_VERSION;
	memcpy ( session->isid, isid, sizeof ( session->isid ) );

	session->portal.tag = PORTAL_GROUP_TAG_UNKNOWN;
}

int iscsiIoctl ( int request, void *data ) 
{
	if ( iscsifd < 0 ) {
		iscsifd = open ( ISCSIDEV, O_RDWR );
		if ( iscsifd < 0 )
			return -1;
	}

	return ( ioctl ( iscsifd, request, data ) );
}

char *determineTableOffset(const unsigned long  map_base)
{
	int                    i       = 0;
	unsigned char          *cur_ptr;
	struct ibft_table_hdr  *ibft_hdr;
	unsigned char          check_sum;
	cur_ptr = (unsigned char*)map_base;

	while((unsigned int)cur_ptr < (map_base+MAP_SIZE))	{
		if (memcmp(cur_ptr, iBFTSTR,strlen(iBFTSTR))) {
			cur_ptr++;
			continue;
		}

		ibft_hdr = (struct ibft_table_hdr *)cur_ptr;
		/* Make sure it's correct version. */
		if (ibft_hdr->revision != iBFT_REV)
			continue;

		/* Make sure that length is valid. */
		if ((unsigned int)(cur_ptr + ibft_hdr->length) < (map_base + MAP_SIZE)) 
		{
			/* Let verify the checksum */
			for (i = 0, check_sum = 0; i < ibft_hdr->length; i++)
				check_sum += cur_ptr[i];

			if (check_sum == 0)
			{
				return ((char *)cur_ptr);
			}
		}
	}
	return NULL;
}

BOOL ifup(const netdev_t* nic)
{
	struct ifreq       ifr;
	struct sockaddr_in sin;
	int                sock = 0;
	char               netmask[MAX_NETMASK_LENGTH];
	char               address[MAX_NETMASK_LENGTH];
	char               g_address[MAX_NETMASK_LENGTH];
	struct rtentry     route;
	struct sockaddr_in sindst,singw;

	memset(netmask, 0, MAX_NETMASK_LENGTH);

	if (0 >= strlen(nic->ifname))
	{
		iscsi_printf("The interface name is empty\n");
	}

	sock = socket(PF_INET,SOCK_DGRAM, 0);

	if (-1 == sock)
	{
		iscsi_printf("Could not create a socket\n");
		return FALSE;
	}

	memset(&ifr, 0, sizeof(struct ifreq));
	memset(&sin, 0, sizeof(struct sockaddr_in));
	strcpy(ifr.ifr_name, nic->ifname);

	sin.sin_family = AF_INET;
	sin.sin_port = 0;

	ipToAddress(&(((u_int32_t *)nic->n_ip)[3]), address);

	inet_aton(address, &(sin.sin_addr));

	memcpy(&ifr.ifr_addr, &sin, sizeof(struct sockaddr));

	iscsi_printf("Starting interface %s with address: %s\n", 
			ifr.ifr_name, inet_ntoa(sin.sin_addr));

	// Set the IP address
	if (0 > ioctl(sock, SIOCSIFADDR, &ifr))
	{
		iscsi_printf("Could not set the IP address!\n");
		close(sock);
		return FALSE;
	}

	// Configure the netmask
	memset(&ifr, 0, sizeof(struct ifreq));
	memset(&sin, 0, sizeof(struct sockaddr_in));
	strcpy(ifr.ifr_name, nic->ifname);

	sin.sin_family = AF_INET;
	sin.sin_port = 0;

	prefixToNetmask(nic->n_net, netmask);
	inet_aton(netmask, &(sin.sin_addr));

	memcpy(&ifr.ifr_netmask, &sin, sizeof(struct sockaddr));

	// Set the netmask
	if (0 > ioctl(sock, SIOCSIFNETMASK, &ifr))
	{
		iscsi_printf("1 Could not set the netmask!\n");
		close(sock);
		return FALSE;
	}

	// Get the current flags
	memset(&ifr, 0, sizeof(struct ifreq));
	strcpy(ifr.ifr_name, nic->ifname);
	if (0 > ioctl(sock, SIOCGIFFLAGS, &ifr))
	{
		iscsi_printf("Could not get flags\n");
		close(sock);
		return FALSE;
	}

	// Set the UP flag
	ifr.ifr_flags |= IFF_UP;

	// Set the new flags, which will bring up the interface
	if (0 > ioctl(sock, SIOCSIFFLAGS, &ifr)) 
	{
		iscsi_printf("Could not start the network device\n");
		close(sock);
		return FALSE;
	}

	//
	// Set gateway
	//
	ipToAddress(&(((u_int32_t *)nic->n_gate)[3]), g_address);

	memset ( &route, 0, sizeof ( route ) );

	sindst.sin_family = AF_INET;
	singw.sin_family = AF_INET;

	sindst.sin_addr.s_addr = INADDR_ANY;
	inet_aton(g_address, &(singw.sin_addr));

	route.rt_dst = *(struct sockaddr *)&sindst; 
	route.rt_gateway = *(struct sockaddr *)&singw;
	route.rt_flags = RTF_GATEWAY | RTF_UP;

	// Get socket 
	if ( sockfd < 0 ) 
	{
		sockfd = socket ( PF_INET, SOCK_STREAM, 0 );
		if ( sockfd < 0 ) 
		{
			iscsi_printf ( "Could not create socket fd\n" );
			close(sock);
			return -1;
		}
	}

	if( ioctl(sockfd, SIOCADDRT, &route ) < 0 )
	{
		if(errno==EEXIST)
		{
			iscsi_printf("Default Gateway already configured\n");
			close(sock);
			return TRUE;
		}
		else
		{
			iscsi_printf("Adding Default Gateway failed\n");
			close(sock);
			return FALSE;
		}

	}

	close(sock);

	return TRUE;
}

int determineOffsets( void *ibft_loc)
{
	struct ibft_table_hdr *ibft_hdr = ibft_loc;
	struct ibft_control *control;
	struct ibft_initiator *initiator = NULL;
	struct ibft_nic *nic0 = NULL, *nic1 = NULL;
	struct ibft_tgt *tgt0 = NULL, *tgt1 = NULL;

	control = ibft_loc + sizeof(*ibft_hdr);
	CHECK_HDR(control, control);

	if (control->initiator_off) {
		initiator = ibft_loc + control->initiator_off;
		CHECK_HDR(initiator, initiator);
	}

	if (control->nic0_off) {
		nic0 = ibft_loc + control->nic0_off;
		CHECK_HDR(nic0, nic);
	}

	if (control->nic1_off) {
		nic1 = ibft_loc + control->nic1_off;
		CHECK_HDR(nic1, nic);
	}

	if (control->tgt0_off) {
		tgt0 = ibft_loc + control->tgt0_off;
		CHECK_HDR(tgt0, target);
	}

	if (control->tgt1_off) {
		tgt1 = ibft_loc + control->tgt1_off;
		CHECK_HDR(tgt1, target);
	}

	nicdev0 = (netdev_t *)malloc(sizeof(netdev_t));
	nicdev1 = (netdev_t *)malloc(sizeof(netdev_t));

	//is NIC 1 enabled?
	if (0x1 & (int)(nic1->hdr.flags))
	{
		//		offset->nic_1.enabled = TRUE;
		nicdev1->enabled = TRUE;

		//is NIC 1 primary?
		if (0x2 & (int)(nic1->hdr.flags))
		{
			nicdev1->primary = TRUE;
		}
		else
		{
			nicdev1->primary = FALSE;
		}

		nicdev1->n_ip    = nic1->ip_addr;
		nicdev1->n_mac   = nic1->mac;
		nicdev1->n_net   = nic1->subnet_mask_prefix;
		nicdev1->n_gate  = nic1->gateway;

	}
	else
	{
		nicdev1->enabled = FALSE;
	}


	//is NIC 0 enabled?
	if (0x1 & (int)(nic0->hdr.flags))
	{
		nicdev0->enabled = TRUE;

		//is NIC 0 primary?
		if (0x2 & (int)(nic0->hdr.flags))
		{
			nicdev0->primary = TRUE;
		}
		else
		{
			nicdev0->primary = FALSE;
		}

		nicdev0->n_ip   = nic0->ip_addr;
		nicdev0->n_mac  = nic0->mac;
		nicdev0->n_net  = nic0->subnet_mask_prefix;
		nicdev0->n_gate = nic0->gateway;

	}
	else
	{
		nicdev0->enabled = FALSE;
	}


	if (FALSE == nicdev0->enabled && FALSE == nicdev1->enabled)
	{
		iscsi_printf("OptionROM did not report either NIC as enabled, assuming eth0\n");
		return -1;
	}
	return 0;
}


BOOL ipToAddress(const u_int32_t* ip, char* address)
{
	u_int32_t a = (((u_int8_t*)ip)[0]);
	u_int32_t b = (((u_int8_t*)ip)[1]);
	u_int32_t c = (((u_int8_t*)ip)[2]);
	u_int32_t d = (((u_int8_t*)ip)[3]);

	sprintf(address, "%u.%u.%u.%u\n", a, b, c, d);
	return TRUE;
}

BOOL prefixToNetmask(u_int32_t prefix, char* netmask)
{
	u_int32_t mask = 0;
	int       index;

	for (index = prefix ; index > 0 ; index--)
	{
		mask = mask << 1;
		mask |= 0x01;
	}

	mask = mask << (32 - prefix);

	sprintf(netmask, "%u.%u.%u.%u\n", 
			((0xFF000000 & mask) >> 24),
			((0x00FF0000 & mask) >> 16),
			((0x0000FF00 & mask) >> 8),
			((0x000000FF & mask)));

	return TRUE;
}
