/*************************************************************************************************
// INTEL CONFIDENTIAL Copyright 2011-2020 Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to the source code
// ("Material") are owned by Intel Corporation or its suppliers or licensors. Title to the Material
// remains with Intel Corporation or its suppliers and licensors. The Material contains trade secrets
// and proprietary and confidential information of Intel or its suppliers and licensors. The Material is
// protected by worldwide copyright and trade secret laws and treaty provisions. No part of the
// Material may be used, copied, reproduced, modified, published, uploaded, posted, transmitted,
// distributed, or disclosed in any way without Intel's prior express written permission.
//
// No license under any patent, copyright, trade secret or other intellectual property right is
// granted to or conferred upon you by disclosure or delivery of the Materials, either expressly, by
// implication, inducement, estoppel or otherwise. Any license under such intellectual property
// rights must be express and approved by Intel in writing.
//*************************************************************************************************/

#ifndef __SS_SENSOR_H__
#define __SS_SENSOR_H__

#include "device.h"

/*! "brief RegWrite
 *    i2c write
 *
 * Arg:
 *    PDEVICE_CONTEXT SensorCtx - context pointer of sensor device
 *    UINT16 Addr - register address
 *    ULONG Data - register data to write
 *    DATA_TOKE_TYPE DataBits - data bits type
 * Return:
 *    NTSTATUS - status
 */
NTSTATUS RegWrite(
    PDEVICE_CONTEXT SensorCtx,
    UINT16 Addr,
    ULONG Data,
    DATA_TOKE_TYPE DataBits
    );

/*! "brief RegWriteExt
*    i2c write
*
* Arg:
*    PDEVICE_CONTEXT SensorCtx - context pointer of sensor device
*    ULONG Addr - register address
*    ULONG Data - register data to write
*    DATA_TOKE_TYPE DataBits - data bits type
*    ADDR_TOKE_TYPE AddrBits - address bits type
*    ULONG I2CIndex          - i2c index
* Return:
*    NTSTATUS - status
*/
NTSTATUS RegWriteExt(
    PDEVICE_CONTEXT SensorCtx,
    ULONG Addr,
    ULONG Data,
    DATA_TOKE_TYPE DataBits,
    ADDR_TOKE_TYPE AddrBits,
    ULONG I2CIndex
    );

/*! "brief RegRead
 *    i2c read
 *
 * Arg:
 *    PDEVICE_CONTEXT SensorCtx - context pointer of sensor device
 *    UINT16 Addr - register address
 *    ULONG* Data - register data to read
 *    DATA_TOKE_TYPE DataBits - data bits type
 * Return:
 *    NTSTATUS - status
 */
NTSTATUS RegRead(
    PDEVICE_CONTEXT SensorCtx,
    UINT16 Addr,
    ULONG *Data,
    DATA_TOKE_TYPE DataBits
    );

/*! "brief Cmd_BulkWriteReg
 *    i2c write
 *
 * Arg:
 *    PDEVICE_CONTEXT SensorCtx - context pointer of sensor device
 *    PSNSR_REG_LIST pInputReg - Registers need to be set, including address and width
 *    DATA_TOKE_TYPE DataBits - data bits type
 * Return:
 *    NTSTATUS - status
 */
NTSTATUS Cmd_BulkWriteReg(
    PDEVICE_CONTEXT SensorCtx,
    PSNSR_REG_LIST pInputReg
    );

/*! "brief Cmd_BulkWriteReg
 *    i2c write
 *
 * Arg:
 *    PDEVICE_CONTEXT SensorCtx - context pointer of sensor device
 *    PSNSR_REG_LIST pInputReg - Registers need to be get, including address and width
 *    PSNSR_REG_LIST pOutputReg - Register data after read
 *    DATA_TOKE_TYPE DataBits - data bits type
 * Return:
 *    NTSTATUS - status
 */
NTSTATUS Cmd_BulkReadReg(
    PDEVICE_CONTEXT SensorCtx,
    PSNSR_REG_LIST pInputReg,
    PSNSR_REG_LIST pOutputReg
    );


/*! "brief RegReadExt
*    i2c read
*
* Arg:
*    PDEVICE_CONTEXT SensorCtx - context pointer of sensor device
*    UINT16 Addr - register address
*    ULONG* Data - register data to read
*    DATA_TOKE_TYPE DataBits - data bits type
*    ADDR_TOKE_TYPE AddrBits - address bits type
*    ULONG I2CIndex          - I2c index
* Return:
*    NTSTATUS - status
*/
NTSTATUS RegReadExt(
    PDEVICE_CONTEXT SensorCtx,
    ULONG Addr,
    ULONG *Data,
    DATA_TOKE_TYPE DataBits,
    ADDR_TOKE_TYPE AddrBits,
    ULONG I2CIndex
    );

/*! "brief RegWriteBufSlow
 *    i2c write buffer
 *
 * Arg:
 *    PDEVICE_CONTEXT pDevExt - context pointer of sensor device
 *    const UINT8* Data - register data to write
 *    ULONG Length - data length
 * Return:
 *    NTSTATUS - status
 */
NTSTATUS RegWriteBufSlow(
    PDEVICE_CONTEXT pDevExt,
    const UINT8* Data,
    ULONG Length
    );

/*! "brief RegWriteQueueSlow
 *    i2c write sensor setting 
 *
 * Arg:
 *    PDEVICE_CONTEXT pDevExt - context pointer of sensor device
 *    const REGS_SETTING *regs - sensor register setting to write
 * Return:
 *    NTSTATUS - status
 */
NTSTATUS RegWriteQueueSlow(
    PDEVICE_CONTEXT SensorCtx,
    const REGS_SETTING *regs
    );

/*! "brief RegWriteQueue
 *    i2c write sensor setting 
 *
 * Arg:
 *    PDEVICE_CONTEXT SensorCtx - context pointer of sensor device
 *    const REGS_SETTING *regs - sensor register setting to write
 * Return:
 *    NTSTATUS - status
 */
NTSTATUS RegWriteQueue(
    PDEVICE_CONTEXT SensorCtx,
    const REGS_SETTING *regs
    );

/*! "brief RegWriteQueueExt
*    i2c write sensor setting
*
* Arg:
*    PDEVICE_CONTEXT SensorCtx - context pointer of sensor device
*    const REGS_SETTING_EXT *regs - sensor register setting to write
*    ULONG I2CIndex  - I2C index
* Return:
*    NTSTATUS - status
*/
NTSTATUS RegWriteQueueExt(
    PDEVICE_CONTEXT SensorCtx,
    const REGS_SETTING_EXT *regs,
    ULONG I2CIndex
    );

/*! "brief SsIvcamGetCalibrationData
 *    iv cam sensor get calibration data 
 *
 * Arg:
 *    PULONG pHandle - device handle
 *    PUINT8 pBuffer - data buffer
 * Return:
 *    NTSTATUS - status
 */
NTSTATUS SsIvcamGetCalibrationData(
    PULONG pHandle,
    PUINT8 pBuffer
    );

/*! "brief SsInitRegistry
 *    init registry for special function
 *
 * Arg:
 *    PDEVICE_CONTEXT SensorCtx - context pointer of sensor device
 */
void SsInitRegistry(
    IN PDEVICE_CONTEXT SensorCtx
    );

/*! "brief SsInitContextCommon
 *    init common context data of sensor device
 *
 * Arg:
 *    PDEVICE_CONTEXT SensorCtx - context pointer of sensor device
 */
void SsInitContextCommon(
    PDEVICE_CONTEXT deviceCtx
    );

/*! "brief SsInitContextCommonPost
 *    init common context data of sensor device
 *
 * Arg:
 *    PDEVICE_CONTEXT SensorCtx - context pointer of sensor device
 */
void SsInitContextCommonPost(
    PDEVICE_CONTEXT deviceCtx
    );


/*! "brief SsSesnsorAcquireGuidIndex
*   Acqueir 1 bit of the Global mask for multiple device instances of the same sensor type
*
* Arg:
*    PDEVICE_CONTEXT SensorCtx - context pointer of sensor device
*    PINT32 pIndex             - Index allocated for the current instance
* Return:
*    NTSTATUS - status
*/
NTSTATUS SsSensorAcquireGuidIndex(
    PDEVICE_CONTEXT deviceCtx
    );

/*! "brief SsSesnsorReleaseGuidIndex
*   Release 1 bit of the Global mask for multiple device instances of the same sensor type
*
* Arg:
*    PDEVICE_CONTEXT SensorCtx - context pointer of sensor device
*    INT32 Index               - Index to be released
* Return:
*    NTSTATUS - status
*/
NTSTATUS SsSensorReleaseGuidIndex(
    PDEVICE_CONTEXT deviceCtx
    );

/*! "brief SsInitExportResTable
 *    initialization of export resolution table
 *
 * Arg:
 *    PDEVICE_CONTEXT SensorCtx - context pointer of sensor device
 * Return:
 *    NTSTATUS - status
 */
NTSTATUS SsInitExportResTable(
    PDEVICE_CONTEXT deviceCtx
    );

/*! "brief SsPowerGpio
 *    power management in gpio part
 *
 * Arg:
 *    PDEVICE_CONTEXT SensorCtx - context pointer of sensor device
 *    BOOLEAN PowerOn - power on or off
 * Return:
 *    NTSTATUS - status
 */
NTSTATUS SsPowerGpio(
    PDEVICE_CONTEXT SensorCtx,
    BOOLEAN PowerOn
    );

/*! "brief Cmd_Idle
 *    enter power idle state
 *
 * Arg:
 *    PDEVICE_CONTEXT SensorCtx - context pointer of sensor device
 * Return:
 *    NTSTATUS - status
 */
NTSTATUS Cmd_Idle(
    PDEVICE_CONTEXT pDevExt
    );

/*! "brief Cmd_Active
 *    enter power active state
 *
 * Arg:
 *    PDEVICE_CONTEXT SensorCtx - context pointer of sensor device
 * Return:
 *    NTSTATUS - status
 */
NTSTATUS Cmd_Active(
    PDEVICE_CONTEXT pDevExt
    );

/*! "brief Cmd_Startup
 *    sensor power start up
 *
 * Arg:
 *    PDEVICE_CONTEXT SensorCtx - context pointer of sensor device
 * Return:
 *    NTSTATUS - status
 */
NTSTATUS Cmd_Startup(
    PDEVICE_CONTEXT pDevExt
    );

/*! "brief Cmd_Shutdown
 *    sensor power shut down
 *
 * Arg:
 *    PDEVICE_CONTEXT SensorCtx - context pointer of sensor device
 * Return:
 *    NTSTATUS - status
 */
NTSTATUS Cmd_Shutdown(
    PDEVICE_CONTEXT pDevExt
    );

/*! "brief Cmd_GetModeData
 *    get mode data
 *
 * Arg:
 *    PDEVICE_CONTEXT SensorCtx - context pointer of sensor device
 *    PULONG Size - data size of mode data need to get
 *    PUINT8 Buffer - buffer pointer
 * Return:
 *    NTSTATUS - status
 */
NTSTATUS Cmd_GetModeData(
    PDEVICE_CONTEXT pDevExt,
    PULONG Size,
    PUINT8 Buffer
    );

/*! "brief Cmd_SetResolution
 *    set sensor resolution
 *
 * Arg:
 *    PDEVICE_CONTEXT SensorCtx - context pointer of sensor device
 *    SNSR_MODE Mode - preview/still/vide mode
 *    ULONG Width - width of resolution 
 *    ULONG Height - height of resolution
 *    LONG frameRate - fps
 * Return:
 *    NTSTATUS - status
 */
NTSTATUS Cmd_SetResolution(
    PDEVICE_CONTEXT pDevExt,
    SNSR_MODE Mode,
    ULONG Width,
    ULONG Height,
    LONG frameRate
    );


/*! "brief Cmd_SetResolutionIndex
 *    set sensor resolution
 *
 * Arg:
 *    PDEVICE_CONTEXT SensorCtx - context pointer of sensor device
 *    SNSR_MODE Mode - preview/still/vide mode
 *    ULONG Index - index of resolution 
 * Return:
 *    NTSTATUS - status
 */
NTSTATUS Cmd_SetResolutionIndex(
    PDEVICE_CONTEXT SensorCtx,
    SNSR_MODE Mode,
    ULONG Index
    );

/*! "brief Cmd_GetSensorInfo
 *    get sensor information
 *
 * Arg:
 *    PDEVICE_CONTEXT SensorCtx - context pointer of sensor device
 *    PSNSR_INFO pInfo - pointer of information buffer
 * Return:
 *    NTSTATUS - status
 */
NTSTATUS Cmd_GetSensorInfo(
    PDEVICE_CONTEXT pDevExt,
    PSNSR_INFO pInfo
    );

/*! "brief Cmd_GetMIPIConfigs
 *    get sensor mipi configuration
 *
 * Arg:
 *    PDEVICE_CONTEXT SensorCtx - context pointer of sensor device
 *    ULONG *Port - mipi csi port
 *    ULONG *LaneNum - mipi lanes number
 *    ULONG *OutputFormat - sensor output format
 * Return:
 *    NTSTATUS - status
 */
void Cmd_GetMIPIConfigs(
    PDEVICE_CONTEXT pDevExt,
    ULONG *Port,
    ULONG *LaneNum,
    ULONG *OutputFormat
    );

/*! "brief Cmd_GetCaps
 *    get sensor information
 *
 * Arg:
 *    PDEVICE_CONTEXT SensorCtx - context pointer of sensor device
 *    SNSR_CAPS **Caps - pointer of capability buffer
 * Return:
 *    NTSTATUS - status
 */
NTSTATUS Cmd_GetCaps (
    PDEVICE_CONTEXT pDevExt,
    SNSR_CAPS **Caps
    );

/*! "brief Cmd_GetResolutions
 *    get sensor resolutions
 *
 * Arg:
 *    PDEVICE_CONTEXT SensorCtx - context pointer of sensor device
 *    SNSR_RESOLUTION_SET **StillRes - still resolution list
 *    SNSR_RESOLUTION_SET **VideoRes - video resolution list
 *    SNSR_RESOLUTION_SET **PreviewRes - preview resolution list
 *    SNSR_RESOLUTION_SET **RawRes - raw resolution list
 * Return:
 *    NTSTATUS - status
 */
NTSTATUS Cmd_GetResolutions (
    PDEVICE_CONTEXT pDevExt,
    SNSR_RESOLUTION_SET **StillRes,
    SNSR_RESOLUTION_SET **VideoRes,
    SNSR_RESOLUTION_SET **PreviewRes,
    SNSR_RESOLUTION_SET **RawRes
    );

/*! "brief Cmd_GetVersion
 *    get sensor version
 *
 * Arg:
 *    PDEVICE_CONTEXT SensorCtx - context pointer of sensor device
 *    SNSR_VERSION_INFO *Version - still resolution list
 * Return:
 *    NTSTATUS - status
 */
NTSTATUS Cmd_GetVersion(
    PDEVICE_CONTEXT pDevExt,
    SNSR_VERSION_INFO *Version
    );

/*! "brief Cmd_GetPld
 *    get sensor pld information
 *
 * Arg:
 *    PDEVICE_CONTEXT SensorCtx - context pointer of sensor device
 *    PACPI_PLD_V2_BUFFER Pld - sensor pld information pointer
 * Return:
 *    NTSTATUS - status
 */
NTSTATUS Cmd_GetPld(
    IN PDEVICE_CONTEXT pDevExt,
    OUT PACPI_PLD_V2_BUFFER Pld
    );

/*! "brief Cmd_GetFrameRate
 *    get sensor frame rate
 *
 * Arg:
 *    PDEVICE_CONTEXT SensorCtx - context pointer of sensor device
 *    PULONG pFps - sensor frame rate, the FPS should be divided by 1000, and we can get the real value.
 * Return:
 *    NTSTATUS - status
 */
NTSTATUS Cmd_GetFrameRate(
    PDEVICE_CONTEXT pDevExt,
    PULONG pFps
    );

/*! "brief Cmd_GetBinningMode
 *    get sensor binning mode
 *
 * Arg:
 *    PDEVICE_CONTEXT SensorCtx - context pointer of sensor device
 *    PULONG pBinning - binning mode
 * Return:
 *    NTSTATUS - status
 */
NTSTATUS Cmd_GetBinningMode(
    PDEVICE_CONTEXT pDevExt,
    PULONG pBinning
    );

/*! "brief Cmd_ByteWriteReg
*    write register with 8bits Data
*
* Arg:
*    PDEVICE_CONTEXT SensorCtx - context pointer of sensor device
*    UINT16 Addr - addr of regsiter
*    UINT16 Data - data to write
* Return:
*    NTSTATUS - status
*/
NTSTATUS Cmd_ByteWriteReg(
    PDEVICE_CONTEXT SensorCtx,
    UINT16 Addr,
    UINT16 Data
    );

/*! "brief Cmd_WriteReg
 *    write register
 *
 * Arg:
 *    PDEVICE_CONTEXT SensorCtx - context pointer of sensor device
 *    UINT16 Addr - addr of regsiter
 *    UINT16 Data - data to write
 * Return:
 *    NTSTATUS - status
 */
NTSTATUS Cmd_WriteReg(
    PDEVICE_CONTEXT SensorCtx,
    UINT16 Addr,
    UINT16 Data
    );

/*! "brief Cmd_ByteReadReg
*    read 8bits data out of register 
*
* Arg:
*    PDEVICE_CONTEXT SensorCtx - context pointer of sensor device
*    UINT16 Addr - addr of regsiter
*    UINT16* pData - data to read
* Return:
*    NTSTATUS - status
*/
NTSTATUS Cmd_ByteReadReg(
    PDEVICE_CONTEXT SensorCtx,
    UINT16 Addr,
    UINT16* pData
    );

/*! "brief Cmd_ReadReg
 *    read register
 *
 * Arg:
 *    PDEVICE_CONTEXT SensorCtx - context pointer of sensor device
 *    UINT16 Addr - addr of regsiter
 *    UINT16* pData - data to read
 * Return:
 *    NTSTATUS - status
 */
NTSTATUS Cmd_ReadReg(
    PDEVICE_CONTEXT SensorCtx,
    UINT16 Addr,
    UINT16* pData
    );

/*! "brief Cmd_StartViewfinder
 *    start streaming of sensor
 *
 * Arg:
 *    PDEVICE_CONTEXT SensorCtx - context pointer of sensor device
 *    ULONG Width - width of resolution
 *    ULONG Height - height of resolution
 * Return:
 *    NTSTATUS - status
 */
NTSTATUS Cmd_StartViewfinder(
    PDEVICE_CONTEXT SensorCtx,
    ULONG Width,
    ULONG Height
    );

/*! "brief Cmd_StopViewfinder
 *    stop streaming of sensor
 *
 * Arg:
 *    PDEVICE_CONTEXT SensorCtx - context pointer of sensor device
 * Return:
 *    NTSTATUS - status
 */
NTSTATUS Cmd_StopViewfinder(
    PDEVICE_CONTEXT SensorCtx
    );

/*! "brief Cmd_GetExtFeature
 *    get external features
 *
 * Arg:
 *    PDEVICE_CONTEXT SensorCtx - context pointer of sensor device
 *    SNSR_EXT_FEATURE_ID feature - feature id
 *    PULONG pValue - data pointer
 * Return:
 *    NTSTATUS - status
 */
NTSTATUS Cmd_GetExtFeature(
    PDEVICE_CONTEXT pDevExt,
    SNSR_EXT_FEATURE_ID feature,
    PULONG pValue
    );

/*! "brief Cmd_SetExtFeature
 *    set external features
 *
 * Arg:
 *    PDEVICE_CONTEXT SensorCtx - context pointer of sensor device
 *    SNSR_EXT_FEATURE_ID feature - feature id
 *    ULONG_PTR pValue - data pointer
 * Return:
 *    NTSTATUS - status
 */
NTSTATUS Cmd_SetExtFeature(
    PDEVICE_CONTEXT SensorCtx,
    SNSR_EXT_FEATURE_ID feature,
    ULONG_PTR Value
    );

/*! "brief Cmd_SetPlatformInfo
*    set external features
*
* Arg:
*    PDEVICE_CONTEXT SensorCtx - context pointer of sensor device
*    PPLAT_INFO PlatformInfo - platform information
* Return:
*    NTSTATUS - status
*/
NTSTATUS Cmd_SetPlatformInfo(
    PDEVICE_CONTEXT pDevExt,
    PPLAT_INFO PlatformInfo
    );

/*! "brief Cmd_SetPos
 *    set focus position
 *
 * Arg:
 *    PDEVICE_CONTEXT SensorCtx - context pointer of sensor device
 *    USHORT FocusPos - focus position
 * Return:
 *    NTSTATUS - status
 */
NTSTATUS Cmd_SetPos(
    PDEVICE_CONTEXT pDevExt,
    USHORT FocusPos
    );

/*! "brief Cmd_GetPos
 *    get focus position
 *
 * Arg:
 *    PDEVICE_CONTEXT SensorCtx - context pointer of sensor device
 *    USHORT *FocusPos - focus position
 * Return:
 *    NTSTATUS - status
 */
NTSTATUS Cmd_GetPos(
    PDEVICE_CONTEXT pDevExt,
    USHORT *FocusPos
    );

/*! "brief Cmd_GetStatus
 *    get focus status
 *
 * Arg:
 *    PDEVICE_CONTEXT SensorCtx - context pointer of sensor device
 *    PULONG Status - focus status
 * Return:
 *    NTSTATUS - status
 */
NTSTATUS Cmd_GetStatus(
    PDEVICE_CONTEXT pDevExt,
    PULONG Status    
    );

/*! "brief Cmd_GetHPStatus
 *    get focus home position status
 *
 * Arg:
 *    PDEVICE_CONTEXT SensorCtx - context pointer of sensor device
 *    PULONG Status - focus status
 * Return:
 *    NTSTATUS - status
 */
NTSTATUS Cmd_GetHPStatus(
    PDEVICE_CONTEXT pDevExt,
    PULONG Status    
    );

/*! "brief Cmd_SetConfig
 *    set focus configuration
 *
 * Arg:
 *    PDEVICE_CONTEXT SensorCtx - context pointer of sensor device
 *    SNSR_VCMDATA* pSnsrVcmData - focus vcm configration data
 * Return:
 *    NTSTATUS - status
 */
NTSTATUS Cmd_SetConfig(
    PDEVICE_CONTEXT pDevExt,
    SNSR_VCMDATA* pSnsrVcmData
    );

/*! "brief Cmd_GetNvmData
 *    get NVM data
 *
 * Arg:
 *    PDEVICE_CONTEXT SensorCtx - context pointer of sensor device
 *    PULONG Size - nvram data size
 *    PUINT8 Buffer - nvram data buffer
 * Return:
 *    NTSTATUS - status
 */
NTSTATUS Cmd_GetNvmData(
    PDEVICE_CONTEXT pDevExt,
    PULONG Size,
    PUINT8 Buffer
    );

/*! "brief Cmd_NvmWrite
 *    write NVM data
 *
 * Arg:
 *    PDEVICE_CONTEXT SensorCtx - context pointer of sensor device
 *    UINT16 Addr - nvram data address
 *    UINT8* pData - nvram data buffer
 *    INT32 Length - nvram data length
 * Return:
 *    NTSTATUS - status
 */
NTSTATUS Cmd_NvmWrite(
    PDEVICE_CONTEXT pDevExt,
    UINT16 Addr,
    UINT8* pData,
    INT32 Length
    );

/*! "brief Cmd_NvmRead
 *    read NVM data
 *
 * Arg:
 *    PDEVICE_CONTEXT SensorCtx - context pointer of sensor device
 *    UINT16 Addr - nvram data address
 *    UINT8* pData - nvram data buffer
 *    INT32 Length - nvram data length
 * Return:
 *    NTSTATUS - status
 */
NTSTATUS Cmd_NvmRead(
    PDEVICE_CONTEXT pDevExt,
    UINT16 Addr,
    UINT8* pData,
    INT32 Length
    );

#if defined(OTM_SENSOR_CONTROL_LOGIC)
/*! "brief QueryControlLogicDeviceInterface
 *    query sensor device interface
 *
 * Arg:
 *    PDEVICE_OBJECT deviceObject - device object
 *    CONTROLLOGIC_SENSOR_INTERFACE* intf - control logic driver interface
 * Return:
 *    NTSTATUS - status
 */
NTSTATUS QueryControlLogicDeviceInterface(
    PDEVICE_OBJECT deviceObject,
    CONTROLLOGIC_SENSOR_INTERFACE* intf
    );

/*! "brief CleanUpSensorInterfaceContext
 *    clean up sensor interface context
 *
 * Arg:
 *    PDEVICE_CONTEXT deviceCtx - context pointer of sensor device
 * Return:
 *    NTSTATUS - status
 */
NTSTATUS CleanUpSensorInterfaceContext(
    PDEVICE_CONTEXT deviceCtx
    );
#endif

#if defined(OTM_SENSOR_CONTROL_LOGIC)
/*! "brief Cmd_GetControlLogicInfo
 *    get control logic information
 *
 * Arg:
 *    PDEVICE_CONTEXT deviceCtx - context pointer of sensor device
 *    PVOID info - information of control logic
 * Return:
 *    NTSTATUS - status
 */
NTSTATUS Cmd_GetControlLogicInfo(
    PDEVICE_CONTEXT deviceCtx,
    PVOID info
    );
#endif

/*! "brief QueryCVFInterface
 *    query CVF device interface
 *
 * Arg:
 *    PDEVICE_OBJECT deviceObject - device object
 *    PCVF_INTERFACE_CAMERA intf - CVF driver interface
 * Return:
 *    NTSTATUS - status
 */
NTSTATUS QueryCVFInterface(
    PDEVICE_OBJECT deviceObject,
    PCVF_INTERFACE_CAMERA_VER_12 intf
);

/*! "brief CleanUpCVFInterfaceContext
 *    clean up CVF interface context
 *
 * Arg:
 *    PDEVICE_CONTEXT deviceCtx - context pointer of CVF device
 * Return:
 *    NTSTATUS - status
 */
NTSTATUS CleanUpCVFInterfaceContext(
    PDEVICE_CONTEXT deviceCtx
);

void ConvertULONGForI2C(
    PDEVICE_CONTEXT SensorCtx,
    UINT32 *Data
    );

/*
  init wpp, replace original wpp init function
*/
void TraceWPPInit(PDRIVER_OBJECT, PUNICODE_STRING);

/*
  uninit wpp
*/
void TraceWPPUninit(PDRIVER_OBJECT);

/*
  replace doTraceMassage() function, no need to include wpp in libs
*/
void TraceWPPLogFn(tag_sensor_trace_level level, const PCHAR fmt, ...);

/*! "brief RegisterForWMIPrivacyModeNotification
 *    register wmi notification for privacy mode
 *
 * Arg:
 *    PDEVICE_CONTEXT DeviceCtx - context pointer of sensor device
 *    GUID WMIGuid - GUID of the WMI event
 *    WMI_NOTIFICATION_CALLBACK Callback - the WMI notification callback need to be called when the WMI event is fired
 * Return:
 *    NTSTATUS - status
 */
NTSTATUS RegisterForWMIPrivacyModeNotification(PDEVICE_CONTEXT DeviceCtx,
    GUID WMIGuid, WMI_NOTIFICATION_CALLBACK Callback);

/*! "brief UnregisterForWMIPrivacyModeNotification
 *    unregister wmi notification for privacy mode
 *
 * Arg:
 *    PDEVICE_CONTEXT DeviceCtx - context pointer of sensor device
 * Return:
 *    VOID
 */
VOID UnregisterForWMIPrivacyModeNotification(PDEVICE_CONTEXT DeviceCtx);

/*! "brief WmiPrivacyModeNotificationCallback
 *    wmi notification callback for privacy mode
 *
 * Arg:
 *    PVOID Wnode - points to the WNODE_EVENT_ITEM structure returned by the driver triggering the event.
 *    PVOID Context - points to the value specified in the Context parameter.
 * Return:
 *    VOID
 */
VOID WMIPrivacyNotificationCallbackForOEM(PVOID Wnode, PVOID Context);

/*! "brief WMIQueryPrivacyStateForOEM
 *    query device state for privacy mode
 *
 * Arg:
 *    PVOID Context - context pointer of sensor device
 *    PVOID PrivacyState - pointer of device state for privacy,
 * Return:
 *    NTSTATUS - status
 */
NTSTATUS WMIQueryPrivacyStateForOEM(PVOID Context, PVOID PrivacyState);

/*! "brief CleanUpWMIPrivacyModeInterface
 *    clean up the WMI interface of privacy mode
 *
 * Arg:
 *    PDEVICE_CONTEXT DeviceCtx - context pointer of sensor device
 * Return:
 *    VOID
 */
VOID CleanUpWMIPrivacyModeInterface(PDEVICE_CONTEXT DeviceCtx);
#endif
