/*******************************************************************************
** INTEL CONFIDENTIAL
**
** Copyright (c) 2020-2023 Intel Corporation All Rights Reserved
**
** This software and the related documents are Intel copyrighted materials,
** and your use of them is governed by the express license under which they
** were provided to you ("License"). Unless the License provides otherwise,
** you may not use, modify, copy, publish, distribute, disclose or transmit
** this software or the related documents without Intel's prior written
** permission.
**
** The source code contained or described herein and all documents related to
** the source code ("Material") are owned by Intel Corporation or its suppliers
** or licensors.
**
** This software and the related documents are provided as is,
** with no express or implied warranties, other than those that are expressly
** stated in the License.
**
*******************************************************************************/

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>

#include "IpfClient.h"

#define JSONSTRING(...) #__VA_ARGS__
#define BUFLEN 4096

void Hello(const char event_data[])
{
	printf("Hello from ContextObject! Event data is %s\n", event_data);
}

// Function to be called in case of provider event.
void CallbackFunction(const char* path, const char* event, void* context)
{
	printf("\nReceived event from %s. Event data: %s\n", path, event);
	if (context) {
		Hello(event);
	}
}

int main(int argc, char** argv)
{
	// Create a buffer to hold results
	size_t maxbuf = BUFLEN;
	char* buf = (char*)calloc(maxbuf, sizeof(char));
	if (buf == NULL) {
		return IPF_ERR_MEMORY;
	}
	// A pointer to the ClientApi object used in subsequent calls to the ClientApi
	void* pIpf = NULL;

// clang-format off
#if (_WIN32)
	static char test_config[] = JSONSTRING({
		"schema_type" : "provider_config",
		"schema_revision" : 1,
		"body" : {
			"local_providers" : [
				"SampleCameraProvider.dll",
				"SampleBatteryProvider.dll"
			],
			"init" : {
				"Company.Org.Battery" :
					{
						"Manufacturer" : "UpdatedBatteryManufacturer", 
						"Model" : "MUO6"
					}
			}
		}
	});
// clang-format on 
	
#else
	static char test_config[] = JSONSTRING({
		"schema_type" : "provider_config",
		"schema_revision" : 1,
		"body" : {
			"local_providers" : [
				"libSampleCameraProvider.so",
				"libSampleBatteryProvider.so"

			],
			"init" : {
				"Company.Org.Battery" :
					{"Manufacturer" : "UpdatedBatteryManufacturer", "Model" : "MUO6"}
			}
		}
	});

#endif

	// Create the ClientApi object
	ipf_err_t ret = IpfCreate(test_config, &pIpf);

	// Get ClientApi Properties
	if (ret == IPF_ERR_OK) {
		size_t buflen = maxbuf;
		ret = IpfGetProperties(pIpf, NULL, buf, &buflen);
	}
	if (ret == IPF_ERR_OK) {
		size_t buflen = maxbuf;
		JsonBeautify(buf, 4, buf, &buflen);
		printf("\nipf.GetProperties(NULL) \n%s\n", buf);
	}

	// Get objects and properties under Platform
	// This can return a large amount of data exceeding the buffer size
	// resulting in IPF_ERR_BUFFERTOOSMALL
	if (ret == IPF_ERR_OK) {
		size_t buflen = maxbuf;
		ret = IpfGetNode(pIpf, "Platform", buf, &buflen);
	}
	if (ret == IPF_ERR_OK) {
		// For simplicity, we do not check the result of JsonBeautify before printing the result,
		// since buf is unchanged when an error occurs and input and output buffers are the same.
		// More robust code could check the result before using the beautified JSON string.
		size_t buflen = maxbuf;
		JsonBeautify(buf, 4, buf, &buflen);
		printf("\nipf.GetNode(\"Platform\") \n%s\n", buf);
	}

	// Get all objects and properties under Camera Provider
	if (ret == IPF_ERR_OK) {
		size_t buflen = maxbuf;
		ret = IpfGetNode(pIpf, "Platform.Extension.Camera", buf, &buflen);
	}
	if (ret == IPF_ERR_OK) {
		size_t buflen = maxbuf;
		JsonBeautify(buf, 4, buf, &buflen);
		printf("\nipf.GetNode(\"Platform.Extension.Camera\") \n%s\n", buf);
	}

	// Get Data Schema for Camera Provider
	if (ret == IPF_ERR_OK) {
		size_t buflen = maxbuf;
		ret = IpfGetSchema(pIpf, "Platform.Extension.Camera", IPF_SCHEMA_TYPE_DATA, buf, &buflen);
	}
	if (ret == IPF_ERR_OK) {
		size_t buflen = maxbuf;
		JsonBeautify(buf, 4, buf, &buflen);
		printf("\nipf.GetSchema(\"Platform.Extension.Camera\") \n%s\n", buf);
	}

	// Query node to get objects and properties at any level in the node tree
	if (ret == IPF_ERR_OK) {
		size_t buflen = maxbuf;
		ret = IpfQueryNode(pIpf, "Platform.Extension.Camera", buf, &buflen);
	}
	if (ret == IPF_ERR_OK) {
		size_t buflen = maxbuf;
		JsonBeautify(buf, 4, buf, &buflen);
		printf("\nipf.QueryNode(\"Platform.Extension.Camera\") \n%s\n", buf);
	}

	// Set fps property of Camera
	if (ret == IPF_ERR_OK) {
		ret = IpfSetValue_int(pIpf, "Platform.Extension.Camera.Fps", 100);
	}

	// Get fps property from Camera
	if (ret == IPF_ERR_OK) {
		size_t buflen = maxbuf;
		ret = IpfGetValue(pIpf, "Platform.Extension.Camera.Fps", buf, &buflen);
	}
	if (ret == IPF_ERR_OK) {
		size_t buflen = maxbuf;
		JsonBeautify(buf, 4, buf, &buflen);
		printf("\nipf.GetValue(\"Platform.Extension.Camera.Fps\") \n%s\n", buf);
	}

	// Get init parameter information from Battery provider
	if (ret == IPF_ERR_OK) {
		size_t buflen = maxbuf;
		ret = IpfGetNode(pIpf, "Platform.Extension.Battery", buf, &buflen);
	}
	if (ret == IPF_ERR_OK) {
		size_t buflen = maxbuf;
		JsonBeautify(buf, 4, buf, &buflen);
		printf("\nipf.GetNode(\"Platform.Extension.Battery\") \n%s\n", buf);
	}

	// Get manufacturer information which is a init parameter
	if (ret == IPF_ERR_OK) {
		size_t buflen = maxbuf;
		ret = IpfGetNode(pIpf, "Platform.Extension.Battery.Manufacturer", buf, &buflen);
	}
	if (ret == IPF_ERR_OK) {
		size_t buflen = maxbuf;
		JsonBeautify(buf, 4, buf, &buflen);
		printf("\nipf.GetNode(\"Platform.Extension.Battery.Manufacturer\") \n%s\n", buf);
	}

	// Set state (enum) property of Battery
	if (ret == IPF_ERR_OK) {
		ret = IpfSetValue_string(pIpf, "Platform.Extension.Battery.State", "charged");
	}

	if (ret == IPF_ERR_OK) {
		BOOL is_ion = TRUE;
		ret = IpfSetValue_bool(pIpf, "Platform.Extension.Battery.State", is_ion);
		assert(ret == IPF_ERR_SETVALUE);
		printf("Set value is not supported");
		// Just to set ret == IPF_ERR_OK to execute next commands
		ret = IpfSetValue_string(pIpf, "Platform.Extension.Battery.State", "charged");
	}

	// Get state from Battery
	if (ret == IPF_ERR_OK) {
		size_t buflen = maxbuf;
		ret = IpfGetValue(pIpf, "Platform.Extension.Battery.State", buf, &buflen);
	}
	if (ret == IPF_ERR_OK) {
		size_t buflen = maxbuf;
		JsonBeautify(buf, 4, buf, &buflen);
		printf("\nipf.GetValue(\"Platform.Extension.Battery.State\") \n%s\n", buf);
	}

	// Get child property value under Battery Provider
	if (ret == IPF_ERR_OK) {
		size_t buflen = maxbuf;
		ret = IpfGetValue(pIpf, "Platform.Extension.Battery.ChildCell.State", buf, &buflen);
	}
	if (ret == IPF_ERR_OK) {
		size_t buflen = maxbuf;
		JsonBeautify(buf, 4, buf, &buflen);
		printf("\nipf.GetValue(\"Platform.Extension.Battery.ChildCell.State\") \n%s\n", buf);
	}

	// Register for an event, passing the address of the context object
	if (ret == IPF_ERR_OK) {
		void* ctx = NULL;
		ret = IpfRegisterEvent(
			pIpf, "Platform.Extension.Battery.BatteryFullyCharged", CallbackFunction, &ctx);
	}

	// Execute a command
	if (ret == IPF_ERR_OK) {
		char cmdobj[] = JSONSTRING({"CommandName" : "value1"});
		size_t buflen = maxbuf;
		ret = IpfExecuteCommand(pIpf, "Platform.Extension.Battery", cmdobj, buf, &buflen);
	}
	if (ret == IPF_ERR_OK) {
		size_t buflen = maxbuf;
		JsonBeautify(buf, 4, buf, &buflen);
		printf("Result of ipf.ExecuteCommand(\"Platform.Extension.Battery, cmdobj\") \n%s\n", buf);
	}

	// Unregister for the event
	if (ret == IPF_ERR_OK) {
		ret = IpfUnregisterEvent(
			pIpf, "Platform.Extension.Battery.BatteryFullyCharged", CallbackFunction);
	}

	char cmd_simulate_plug[] = JSONSTRING({
		"SimulatePlug" :
			{"InstanceName" : "ChildBatteryProvider", "Manufacturer" : "Intel", "Model" : "I123X"}
	});

	char cmd_simulate_unplug[] =
		JSONSTRING({"SimulateUnplug" : {"InstanceName" : "ChildBatteryProvider"}});

	// Execute a command
	if (ret == IPF_ERR_OK) {
		size_t buflen = maxbuf;
		ret = IpfExecuteCommand(pIpf, "Platform.Extension.Battery", cmd_simulate_plug, buf, &buflen);
	}
	if (ret == IPF_ERR_OK) {
		size_t buflen = maxbuf;
		JsonBeautify(buf, 4, buf, &buflen);
		printf("ipf.ExecuteCommand(\"Platform.Extension.Battery, cmd_simulate_plug) = \n%s\n", buf);
	}

	// Call child provider's execute command
	if (ret == IPF_ERR_OK) {
		size_t buflen = maxbuf;
		ret = IpfExecuteCommand(pIpf, "Platform.Extension.Battery.ChildBatteryProvider", cmd_simulate_plug, buf, &buflen);
	}
	if (ret == IPF_ERR_OK) {
		size_t buflen = maxbuf;
		JsonBeautify(buf, 4, buf, &buflen);
		printf("ipf.ExecuteCommand(\"Platform.Extension.Battery.ChildBatteryProvider, cmd_simulate_plug) = \n%s\n", buf);
	}

	// Execute command creates a child provider with "ChildBatteryProvider" instance name
	if (ret == IPF_ERR_OK) {
		size_t buflen = maxbuf;
		ret = IpfExecuteCommand(pIpf, "Platform.Extension.Battery", cmd_simulate_unplug, buf, &buflen);
	}
	if (ret == IPF_ERR_OK) {
		size_t buflen = maxbuf;
		JsonBeautify(buf, 4, buf, &buflen);
		printf("ipf.ExecuteCommand(\"Platform.Extension.Battery, cmd_simulate_unplug) = \n%s\n", buf);
	}

	// Error Reporting and Cleanup
	if (ret != IPF_ERR_OK) {
		printf("Error! %d, %s", ret, ipf_ef_error_str(ret));
	}
	if (pIpf) {
		IpfDestroy(pIpf);
	}
	free(buf);
	return ret;
}