﻿/*******************************************************************************
** INTEL CONFIDENTIAL
**
** Copyright (c) 2022-2023 Intel Corporation All Rights Reserved
**
** This software and the related documents are Intel copyrighted materials,
** and your use of them is governed by the express license under which they
** were provided to you ("License"). Unless the License provides otherwise,
** you may not use, modify, copy, publish, distribute, disclose or transmit
** this software or the related documents without Intel's prior written
** permission.
**
** The source code contained or described herein and all documents related to
** the source code ("Material") are owned by Intel Corporation or its suppliers
** or licensors.
**
** This software and the related documents are provided as is,
** with no express or implied warranties, other than those that are expressly
** stated in the License.
**
*******************************************************************************/

using System;
using System.Runtime.InteropServices;
using Ipf;
using Newtonsoft.Json.Linq;

namespace Intel.Ipf.EfSampleApp
{
	internal static class Program
	{
		// An object that can be passed to RegisterEvent as context.
		// A pointer to this object will be passed into the SampleEventCallback
		private class ContextObject
		{
			public void Hello(string eventData)
			{
				Console.WriteLine("Hello from ContextObject! Event data is " + eventData);
			}
		}

		// The EventCallbackDelegate that is passed into RegisterEvent and UnregisterEvent
		private static void SampleEventCallback(string eventName, string eventData, IntPtr context)
		{
			Console.WriteLine("Received event from " + eventName + ". Event data: " + eventData);

			// Convert the context pointer into a GCHandle, then convert the handle into our object
			var handle = GCHandle.FromIntPtr(context);
			var contextObject = handle.Target as ContextObject;

			// Call a function on our context object
			contextObject?.Hello(eventData);
		}

		// Create an instance of the callback delegate so that
		// it doesn't get garbage collected
		private static readonly EventCallbackDelegate callbackInstance = SampleEventCallback;

		private static void ConsolePrint(string field, JToken value)
		{
			if (value == null)
			{
				Console.WriteLine("Unexpected null value : ");
				Console.WriteLine(field);
			}
			else
			{
				Console.WriteLine(field);
				Console.WriteLine(value.ToString());
			}
			Console.WriteLine();
		}

		public static void Main()
		{
			string config = "";
			try
			{
				if (RuntimeInformation.IsOSPlatform(OSPlatform.Windows))
				{
					config = @"
                    {
                        ""schema_type"": ""provider_config"",
                        ""schema_revision"": 1,
                        ""body"" : {
                            ""local_providers"": [
                                    ""SampleCameraProvider.dll"",
                                    ""SampleBatteryProvider.dll""
                                 ],
							""init"": {
								""Company.Org.Battery"":
									{
										""Manufacturer"": ""UpdatedBatteryManufacturer"",
										""Model"": ""MUO6""
									}
								}
							}
                    }
                    ";
				}
				else if (RuntimeInformation.IsOSPlatform(OSPlatform.Linux))
				{
					config = @"
                    {
                        ""schema_type"": ""provider_config"",
                        ""schema_revision"": 1,
                        ""body"" : {
                            ""local_providers"": [
                                    ""libSampleCameraProvider.so"",
                                    ""libSampleBatteryProvider.so""
                                 ],
							""init"": {
                                ""Company.Org.Battery"":
									{
                                    ""Manufacturer"": ""UpdatedBatteryManufacturer"",
									""Model"": ""MUO6""

                                    }
                            }
                        }
                    }
                    ";
				}

				var providerConfig = JObject.Parse(config);

				// Initialize the ClientApi with the JSON config
				using (IIpfEfClient ipf = new ClientApiJson(providerConfig))
				{

					// Get ClientApi Properties
					JToken node = ipf.GetProperties();
					ConsolePrint("ipf.GetProperties()", node);

					// Get all objects and properties under Platform
					node = ipf.GetNode("Platform");
					ConsolePrint("ipf.GetNode(\"Platform\")", node);

					// Get all objects and properties under Camera Provider
					node = ipf.GetNode("Platform.Extension.Camera");
					ConsolePrint("ipf.GetNode(\"Platform.Extension.Camera\")", node);

					// Get Data Schema for Camera Provider
					JObject schema = ipf.GetSchema("Platform.Extension.Camera", SchemaType.DATA);
					ConsolePrint("ipf.GetSchema(\"Platform.Extension.Camera\")", schema);

					// Query node to get objects and properties at any level in the node tree
					JArray queryResult = ipf.QueryNode("Platform.Extension.Camera");
					ConsolePrint("ipf.QueryNode(\"Platform.Extension.Camera\")", queryResult);

					// Set fps property of Camera
					ipf.SetValue("Platform.Extension.Camera.Fps", 100);
					JToken updatedFps = ipf.GetValue("Platform.Extension.Camera.Fps");
					ConsolePrint("ipf.GetValue(\"Platform.Extension.Camera.Fps\")", updatedFps);

					// Get init parameter information from Battery provider
					node = ipf.GetNode("Platform.Extension.Battery");
					ConsolePrint("ipf.GetNode(\"Platform.Extension.Battery\")", node);

					// Get manufacturer information which is a init parameter
					node = ipf.GetNode("Platform.Extension.Battery.Manufacturer");
					ConsolePrint("ipf.GetNode(\"Platform.Extension.Battery.Manufacturer\")", node);

					// Set state (enum) property of Battery
					ipf.SetValue("Platform.Extension.Battery.State", "charged");

					try
					{
						// Set state (enum) property of Battery
						ipf.SetValue("Platform.Extension.Battery.IsIonBattery", true);
					}
					catch (IpfException)
					{
						Console.WriteLine("Set value is not supported");
					}

					// Get the "state" property from the Battery
					JToken updatedState = ipf.GetValue("Platform.Extension.Battery.State");
					ConsolePrint("ipf.GetValue(\"Platform.Extension.Battery.State\")", updatedState);

					// Get child property value under Battery Provider
					JToken childState = ipf.GetValue("Platform.Extension.Battery.ChildCell.State");
					ConsolePrint("ipf.GetValue(\"Platform.Extension.Battery.ChildCell.State\")", childState);

					// Create a context object and get a handle to it from the
					// garbage collector.
					//
					// Calling Alloc() prevents the object from being garbage collected.
					//
					// We must free the handle after unregistering the event or the
					// object will be leaked.
					var contextObject = new ContextObject();
					var contextHandle = GCHandle.Alloc(contextObject);

					// Register for an event.  Passing a context is optional
					ipf.RegisterEvent("Platform.Extension.Battery.BatteryFullyCharged",
						callbackInstance, GCHandle.ToIntPtr(contextHandle));

					// Execute a command on the battery provider
					const string CMD = @"
                     {
                         ""CommandName"": ""value1""
                     }
                 ";
					var commandObject = JObject.Parse(CMD);
					JObject result = ipf.ExecuteCommand("Platform.Extension.Battery", commandObject);
					ConsolePrint("ipf.ExecuteCommand(\"Platform.Extension.Battery, commandObject\")", result);

					// Unregister for the event
					ipf.UnregisterEvent("Platform.Extension.Battery.BatteryFullyCharged",
						callbackInstance);

					const string CMD_SIMULATE_PLUG = @"
					{
						""SimulatePlug"" : {
							 ""InstanceName"": ""ChildBatteryProvider"",
							 ""Manufacturer"" : ""Intel"",
							 ""Model"" : ""I123X""
						}
					}
					";

					const string CMD_SIMULATE_UNPLUG = @"
					{
						""SimulateUnplug"" :{ 
							""InstanceName"": ""ChildBatteryProvider""
						}
					}
					";

					// Execute command creates a child provider with "ChildBatteryProvider" instance name
					var cmdSimPlug = JObject.Parse(CMD_SIMULATE_PLUG);
					result = ipf.ExecuteCommand("Platform.Extension.Battery", cmdSimPlug);
					ConsolePrint("ipf.ExecuteCommand(\"Platform.Extension.Battery, cmd_sim_plug\")", result);

					// Call child provider's execute command
					result = ipf.ExecuteCommand("Platform.Extension.Battery.ChildBatteryProvider", cmdSimPlug);
					ConsolePrint("ipf.ExecuteCommand(\"Platform.Extension.Battery.ChildBatteryProvider, cmd_sim_plug\")", result);

					var cmdSimUnplug = JObject.Parse(CMD_SIMULATE_UNPLUG);
					result = ipf.ExecuteCommand("Platform.Extension.Battery", cmdSimUnplug);
					ConsolePrint("ipf.ExecuteCommand(\"Platform.Extension.Battery, cmd_sim_unplug\")", result);

					// Free the context handle ONLY after unregistering for the event.
					// If the event is raised after the context_handle is freed, it will
					// cause a System.NullReferenceException
					contextHandle.Free();
				}
			}

			// Catch IPF Exceptions
			catch (IpfException e)
			{
				Console.WriteLine($"Got {e.GetType()} with error {e.error} when accessing the namespace:");
				Console.WriteLine(e.Message);
			}
		}
	}
}
