/*******************************************************************************
** INTEL CONFIDENTIAL
**
** Copyright (c) 2020-2023 Intel Corporation All Rights Reserved
**
** This software and the related documents are Intel copyrighted materials,
** and your use of them is governed by the express license under which they
** were provided to you ("License"). Unless the License provides otherwise,
** you may not use, modify, copy, publish, distribute, disclose or transmit
** this software or the related documents without Intel's prior written
** permission.
**
** The source code contained or described herein and all documents related to
** the source code ("Material") are owned by Intel Corporation or its suppliers
** or licensors.
**
** This software and the related documents are provided as is,
** with no express or implied warranties, other than those that are expressly
** stated in the License.
**
*******************************************************************************/

#include <iostream>

#include "IpfClient.h"
#include "nlohmann/json.hpp"

using Json = nlohmann::json;

// An object that can be passed to RegisterEvent as context.
// A pointer to this object will be passed into the SampleEventCallback
class ContextObject
{
public:
	void Hello(std::string event_data)
	{
		std::cout << "Hello from ContextObject! Event data is " << event_data << std::endl;
	}
};

// Function to be called in case of provider event.
void CallbackFunction(const char* path, const char* event, void* context)
{
	std::cout << "Received event from " << path << ". Event data: " << event << std::endl;
	if (context) {
		static_cast<ContextObject*>(context)->Hello(event);
	}
}

static void ConsolePrint(const std::string& field, const std::string& value)
{
	std::cout << field << std::endl << value << std::endl << std::endl;
}

#if (_WIN32)
static std::string test_config = R"(
		{
			"schema_type": "provider_config",
			"schema_revision": 1,
			"body": {
				"local_providers": [
						"SampleCameraProvider.dll",
						"SampleBatteryProvider.dll"
				],	
				"init": {
					"Company.Org.Battery":
						{                           
							"Manufacturer": "UpdatedBatteryManufacturer",
							"Model": "MUO6"
						}
				}
			}
		}
		)";

#else
static std::string test_config = R"(
		{
			"schema_type": "provider_config",
			"schema_revision": 1,
			"body": {
				"local_providers": [
						"libSampleCameraProvider.so",
						"libSampleBatteryProvider.so"
				],
				"init": {
					"Company.Org.Battery":
						{                           
							"Manufacturer": "UpdatedBatteryManufacturer"
							"Model": "MUO6"
						}
				}
			}
		}
		)";

#endif

int main()
{
	int ret = 0;
	//
	// The main object is not declared inside the try-catch block because if an
	// exception is thrown it will require destruction of the object which may hang
	// with no indication of the reason, as it will never reach the catch blocks.
	//
	std::unique_ptr<Ipf::ClientApiJson> ipf = nullptr;

	try {
		// Load and initialize providers
		ipf = std::make_unique<Ipf::ClientApiJson>(Json::parse(test_config));

		// Get ClientApi Properties
		auto jsonStream = ipf->GetProperties();
		ConsolePrint("ipf->GetProperties()", jsonStream.dump(4));

		// Get objects and properties under Platform
		jsonStream = ipf->GetNode("Platform");
		ConsolePrint("ipf->GetNode(\"Platform\")", jsonStream.dump(4));

		// Get all objects and properties under Camera Provider
		jsonStream = ipf->GetNode("Platform.Extension.Camera");
		ConsolePrint("ipf->GetNode(\"Platform.Extension.Camera\")", jsonStream.dump(4));

		// Get Data Schema for Camera Provider
		auto jsonSchema = ipf->GetSchema("Platform.Extension.Camera", Ipf::SchemaType::DATA);
		ConsolePrint("ipf->GetSchema(\"Platform.Extension.Camera\")", jsonSchema.dump(4));

		// Query node to get objects and properties at any level in the node tree
		auto queryResult = ipf->QueryNode("Platform.Extension.Camera");
		ConsolePrint("ipf->QueryNode(\"Platform.Extension.Camera\")", queryResult.dump(4));

		// Set fps property of Camera
		ipf->SetValue("Platform.Extension.Camera.Fps", 100);

		// Get fps property from Camera
		int updatedFps = ipf->GetValue("Platform.Extension.Camera.Fps");
		ConsolePrint(
			"ipf->GetValue(\"Platform.Extension.Camera.Fps\")", std::to_string(updatedFps));

		// Get init parameter information from Battery provider
		jsonStream = ipf->GetNode("Platform.Extension.Battery");
		ConsolePrint("ipf->GetNode(\"Platform.Extension.Battery\")", jsonStream.dump(4));

		// Get manufacturer information which is a init parameter
		jsonStream = ipf->GetNode("Platform.Extension.Battery.Manufacturer");
		ConsolePrint(
			"ipf->GetNode(\"Platform.Extension.Battery.Manufacturer\")", jsonStream.dump(4));

		// Set state (enum) property of Battery
		ipf->SetValue("Platform.Extension.Battery.State", "charged");

		// Try to set IsIonBattery property
		// This will fail because IsOnBattery is a read-only property and there is no
		// SetValue(path, bool) implmented in the provider to handle the request
		try {
			ipf->SetValue("Platform.Extension.Battery.IsIonBattery", true);
		}
		catch (const Ipf::ProviderException& ex) {
			std::cout << "As expected, SetValue is not supported for "
						 "Platform.Extension.Battery.IsIonBattery "
					  << std::endl;
			std::cout << "Error message is: " << ex.what() << std::endl << std::endl;
		}

		// Get state from Battery
		auto updatedState = ipf->GetValue("Platform.Extension.Battery.State");
		ConsolePrint("ipf->GetValue(\"Platform.Extension.Battery.State\")", updatedState);

		// Get child property value under Battery Provider
		auto childState = ipf->GetValue("Platform.Extension.Battery.ChildCell.State");
		ConsolePrint("ipf->GetValue(\"Platform.Extension.Battery.ChildCell.State\")", childState);

		// Register for an event, passing the address of the context object
		ipf->RegisterEvent("Platform.Extension.Battery.BatteryFullyCharged", CallbackFunction);

		// Execute a command
		Json cmdobj = {{"CommandName", "value1"}};
		auto result = ipf->ExecuteCommand("Platform.Extension.Battery", cmdobj);
		ConsolePrint(
			"Result of ipf->ExecuteCommand(\"Platform.Extension.Battery, cmdobj\")",
			result.dump(4));

		// Unregister for the event
		ipf->UnregisterEvent("Platform.Extension.Battery.BatteryFullyCharged", CallbackFunction);

		Json cmd_simulate_plug = {
			{"SimulatePlug",
			 {{"InstanceName", "ChildBatteryProvider"},
			  {"Manufacturer", "Intel"},
			  {"Model", "I123X"}}}};

		Json cmd_simulate_unplug = {{"SimulateUnplug", {{"InstanceName", "ChildBatteryProvider"}}}};

		// Execute command creates a child provider with "ChildBatteryProvider" instance name
		result = ipf->ExecuteCommand("Platform.Extension.Battery", cmd_simulate_plug);
		ConsolePrint(
			"ipf->ExecuteCommand(\"Platform.Extension.Battery, cmd_simulate_plug = \")",
			result.dump(4));

		// Call child provider's execute command
		result = ipf->ExecuteCommand(
			"Platform.Extension.Battery.ChildBatteryProvider", cmd_simulate_plug);
		ConsolePrint(
			"ipf->ExecuteCommand(\"Platform.Extension.Battery.ChildBatteryProvider , "
			"cmd_simulate_plug = \")",
			result.dump(4));

		result = ipf->ExecuteCommand("Platform.Extension.Battery", cmd_simulate_unplug);
		ConsolePrint(
			"ipf->ExecuteCommand(\"Platform.Extension.Battery , cmd_simulate_unplug = \")",
			result.dump(4));
	}

	// Catch IPF exceptions
	catch (const Ipf::IpfException& e) {
		std::cerr << "Got IpfException when accessing the namespace: " << e.what() << std::endl;
		ret = -1;
	}

	catch (const std::exception& e) {
		std::cerr << "Got standard exception: " << e.what() << std::endl;
		ret = -1;
	}

	// Catchall
	catch (...) {
		std::cerr << "Unexpected exception" << std::endl;
		ret = -1;
	}

	return ret;
}
