<#
INTEL CONFIDENTIAL
Copyright (C) 2024 Intel Corporation All Rights Reserved.

The source code contained or described herein and all documents
related to the source code ("Material") are owned by Intel Corporation
or its suppliers or licensors. Title to the Material remains with
Intel Corporation or its suppliers and licensors. The Material
contains trade secrets and proprietary and confidential information of
Intel or its suppliers and licensors. The Material is protected by
worldwide copyright and trade secret laws and treaty provisions. No
part of the Material may be used, copied, reproduced, modified,
published, uploaded, posted, transmitted, distributed, or disclosed in
any way without Intel's prior express written permission.

No license under any patent, copyright, trade secret or other
intellectual property right is granted to or conferred upon you by
disclosure or delivery of the Materials, either expressly, by
implication, inducement, estoppel or otherwise. Any license under such
intellectual property rights must be express and approved by Intel in
writing.

#>

#Requires -RunAsAdministrator

param (
    [switch] $Information,
    [switch] $Install,
    [switch] $Uninstall,
    [switch] $Verbose ,
    [switch] $Silent
)

$scriptVersion = "2538.8.4.0"
$pnpUtil = "C:\Windows\System32\pnputil.exe"

class Component {
    
    [ValidateNotNull()][string]$DisplayName
    [ValidateNotNullOrEmpty()][string]$SearchName
    [ValidateNotNullOrEmpty()][string]$InfName
    [ValidateNotNull()]       [string]$ExtensionGuid
    [string]$OEMInfName
    [string]$Version

    Component($DisplayName, $SearchName, $InfName, $ExtensionGuid) {
        if ($DisplayName -eq "") {
            $this.DisplayName = $SearchName
        }
        else {
            $this.DisplayName = $DisplayName
        }
        $this.SearchName = $SearchName
        $this.InfName = $InfName
        $this.ExtensionGuid = $ExtensionGuid
        $this.OEMInfName = ""
        $this.Version = ""
    }
}

# by design Powershell adds non-discarded output stream values to the return value of the function
# so it's not recommended to use Write-Output in the functions that return values
# caller then should filter out the output stream values from the return value
# when Write-Host is not polluting output stream but it's could not be used for redirection
# So solution is to aggregate the output and print it in the end of the script
# read more here https://devblogs.microsoft.com/scripting/understanding-streams-redirection-and-write-host-in-powershell/
class ActionResult {
    [int]$ReturnCode
    [System.Collections.Generic.List[string]]$InformationBefore
    [System.Collections.Generic.List[string]]$InformationAfter
    [System.Collections.Generic.List[PSCustomObject]]$Table

    ActionResult() {
        $this.ReturnCode = 0
        $this.Table = New-Object System.Collections.Generic.List[PSCustomObject]
        $this.InformationBefore = New-Object System.Collections.Generic.List[string]
        $this.InformationAfter = New-Object System.Collections.Generic.List[string]
    }

    [void] AddInformationBefore($info) {
        $this.InformationBefore.Add($info)
    }

    [void] AddTableRow($row) {
        $this.Table.Add($row)
    }

    [void] AddInformationAfter($info) {
        $this.InformationAfter.Add($info)
    }
}

$SOL_PATTERN = "Intel(R) Active Management Technology - SOL"

$components = @(
    [Component]::new("", "Intel(R) Management Engine Interface #1", "MEI\win10\heci.inf", "")
    , [Component]::new("Intel(R) SOL LMS Extension", "Intel.+SOL LMS Extension", "SOL_LMS_Extension\SOLLMSExtension.inf", "72522ea3-94a8-4184-92c8-88769998a634")
    , [Component]::new("", $SOL_PATTERN, "SOL\mesrl.inf", "")
    , [Component]::new("", "Intel(R) Management and Security Application Local Management", "LMS\LMS.inf", "")
    , [Component]::new("", "Intel(R) Management Engine WMI Provider", "WMIProvider\MEWMIProv.inf", "")
    , [Component]::new("Intel(R) Wireless Manageability Driver extension", "Intel.+Wireless Manageability Driver extension", "wiman_wlan_extension\wiman_wlan_extension.inf", "babe2782-eb1b-46a7-871a-f043d3200542")
    , [Component]::new("", "Intel(R) Wireless Manageability", "WiMan\WiMan.inf", "")
)

$returnCode_SUCCESS = 0
# $returnCode_ADMIN_REQUIRED = 1
$returnCode_GENERAL_FAILURE = 2
$returnCode_INSTALLATION_FAILED = 3
$returnCode_UNINSTALLATION_FAILED = 4
$returnCode_NO_INF_FILES = 5
$returnCode_REBOOT_REQUIRED = 6


function PrintString {
    param(
        [Parameter(Mandatory = $true)] [string]$str
    )
    if (-not $Silent) {
        Write-Output $str
    }
}

function PrintVerbose {
    param(
        [Parameter(Mandatory = $true)] [string]$str
    )
    if ($Verbose) {
        Write-Output $str
    }
}

function Exit-Script {
    param(
        [Parameter(Mandatory = $true)] [int]$exitCode
    )
    PrintString "[!] Exit with Errorlevel $exitCode"
    exit $exitCode
}

# Check params compatibility
if ($Verbose -and $Silent) {
    Write-Output "Error: Verbose and Silent flags cannot be used together."
    $Silent = $false
    Exit-Script $returnCode_GENERAL_FAILURE
}

$action_INFORMATION = 'Information'
$action_INSTALL = 'Install'
$action_UNINSTALL = 'Uninstall'

$action = $action_INFORMATION

$actionCount = 0
if ($Information) { 
    $action = $action_INFORMATION
    $actionCount++ 
}
if ($Install) { 
    $action = $action_INSTALL
    $actionCount++
}
if ($Uninstall) { 
    $action = $action_UNINSTALL
    $actionCount++ 
}

if ($actionCount -gt 1) {
    PrintString "Error: Only one action could be specified."
    Exit-Script $returnCode_GENERAL_FAILURE
}

if ($actionCount -eq 0) {
    PrintString "Error: At least one action should be specified."
    Exit-Script $returnCode_GENERAL_FAILURE
}


$pnpUtilErrorCode_ERROR_NO_MORE_ITEMS = 259
$pnpUtilErrorCode_ERROR_SUCCESS_REBOOT_INITIATED = 1641
$pnpUtilErrorCode_ERROR_SUCCESS_REBOOT_REQUIRED = 3010
$pnpUtilErrorCode_ERROR_FILE_NOT_INSTALLED_INF = -536870340
$pnpUtilErrorCode_ERROR_DRIVER_MATCHING_INF_NOT_FOUND = 1168


function Show-Header {
    PrintString "Intel(R) Management Engine Component Installer v$scriptVersion"
}

# Get-Extensions-Information, GetIntelExtensionInfs, Test-Candidate, InfBelongsToUs, 
#
# Why we need this set of the functions? 
# Our goal is to gather the information about the drivers installed in the system,
# but first we need somehow find them. 
# Most obvious way is to search by the Friendly name.
# And this is good approach since Friendly name tends to be unique identifier 
# given the fact that we prefix our names with Intel (R) string.
# Here comes the problem: Extension INF files doesn't have Friendly name property 
# which makes it impossible to search by it. Moreover another possible candidate -
# Extension GUID is not unique too.
#
# So we need to solve it hard way:
# 1. List extension installed INFs originated by Intel - take last version
# 2. Search original INF file for some specific pattern (Service name, Display name, etc)

function GetIntelExtensionInfs {
    Get-WindowsDriver -Online -All | 
    Where-Object -FilterScript { $_.ProviderName -eq 'Intel' -and $_.ClassName -eq 'Extension' }  |
    Select-Object Driver, OriginalFileName, Version |
    Sort-Object -Descending -Property Version
}

function InfBelongsToUs {
    param (
        [Parameter(Mandatory = $true)] [string]$infName,
        [Parameter(Mandatory = $true)] [string]$extensionGuid,
        [Parameter(Mandatory = $true)] [string]$searchName
    )
    $extensionMatch = Select-String -Quiet -SimpleMatch -Path $infName -Pattern "$extensionGuid"
    $nameMatch = Select-String -Quiet -Path $infName -Pattern "$searchName"
    return $extensionMatch -and $nameMatch
}

function Test-Candidate() {
    param(
        [Parameter(Mandatory = $true)] [System.Object]$candidate 
    )
    foreach ($component in $components) {
        if ($component.ExtensionGuid -eq "") {
            continue
        }
        if ($component.Version -ne "") {
            continue
        }
        if (InfBelongsToUs -infName $candidate.OriginalFileName -extensionGuid $component.ExtensionGuid -searchName $component.SearchName) {
            $component.OEMInfName = $candidate.Driver
            $component.Version = $candidate.Version
            return
        }
    }
}

function Get-ExtensionsInformation {
    $intelExtensions = GetIntelExtensionInfs
    foreach ($candidate in $intelExtensions) {
        Test-Candidate $candidate
    }    
}

function Invoke-Devices-Scan {
    $pnpUtilOutput = & $pnpUtil /scan-devices 
    $errorCode = $LASTEXITCODE
    if ((-Not $?) -or ($errorCode -ne 0)) {
        return 1
    }
    return 0
}

function Get-Latest-Driver {
    param(
        [Parameter(Mandatory = $true)] [System.Object]$friendlyName
    )
    $resultsTable = New-Object System.Collections.Generic.List[System.Object]

    Get-PnpDevice -ErrorAction Stop -FriendlyName $friendlyName | Where-Object { $_.Present -eq $true } | ForEach-Object {
        $deviceStatus = $_.Status
        $errorCode = ""
        $driverVersion = (Get-PnpDeviceProperty -InputObject $_ -KeyName 'DEVPKEY_Device_DriverVersion').Data
        if ($deviceStatus -eq "Error") {
            $errorCode = $_.Problem
        }
        $resultsTable.Add(
            [PSCustomObject]@{ Status = $_.Status; Version = $driverVersion; ErrorCode = $errorCode }
        )
    }
    return $resultsTable |  Sort-Object -Descending -Property Version | Select-Object -First 1
}

function Show-Information {

    $actionResult = [ActionResult]::new()
    $actionResult.AddInformationBefore("[+] Scanning Devices\Drivers")

    if (Invoke-Devices-Scan -ne 0) {
        $actionResult.ReturnCode = $returnCode_GENERAL_FAILURE
        return $actionResult
    }
    $actionResult.AddInformationBefore("[+] Driver Information:")
    
    Update-Components-Information

    foreach ( $component in $components ) {
        $deviceStatus = ""
        $errorCode = ""
        $verboseDetails = ""
        try {
            if ($component.Version -eq "") {
                $latestDriver = Get-Latest-Driver $component.SearchName
                if ($null -eq $latestDriver) {
                    $driverVersion = "Not installed"
                }
                else {
                    $deviceStatus = $latestDriver.Status
                    $driverVersion = $latestDriver.Version
                    if ($deviceStatus -eq "Error") {
                        $errorCode = $latestDriver.ErrorCode
                    }
                }
            }
            else {
                $deviceStatus = "OK"
                $driverVersion = $component.Version
                $verboseDetails = "Extension INF"
            }
            
        }
        catch [Microsoft.PowerShell.Cmdletization.Cim.CimJobException] {
            $driverVersion = "Not installed"
        }
        catch {
            $driverVersion = "Not installed"
            $verboseDetails = $PSItem.Exception
        }
        if (-not $Verbose) {
            $verboseDetails = ""
        }
        $actionResult.AddTableRow(
            [PSCustomObject]@{Component = $component.DisplayName; Version = $driverVersion; Status = $deviceStatus; ErrorCode = $errorCode; Details = $verboseDetails }
        )
    }

    $actionResult.ReturnCode = $returnCode_SUCCESS
    return $actionResult
}

function Install-Software {

    $actionResult = [ActionResult]::new()
    $actionResult.AddInformationBefore("[+] Starting installation of CSE Software...")
    $actionResult.ReturnCode = $returnCode_SUCCESS
    $rebootRequired = $false

    $componentsFound = 0
    $componentsFailed = 0
    foreach ( $component in $components ) {
        if ( ! (Test-Path $component.InfName) ) {
            if ($Verbose) {
                $actionResult.AddTableRow(
                    [PSCustomObject]@{Component = $component.DisplayName; Status = ""; ErrorCode = ""; Details = "INF is absent" }
                )
            }
            continue
        }
        $verboseDetails = ""
        $componentsFound += 1
        $pnpUtilOutput = & $pnpUtil /add-driver $($component.InfName) /install
        $pnpUtilOutput = "Inf:" + $component.InfName + " " + $pnpUtilOutput
        $errorCode = $LASTEXITCODE
        if ((-Not $?) -or ($errorCode -ne 0)) {
            $deviceStatus = 'Failed'
            switch ( $errorCode ) {
                $pnpUtilErrorCode_ERROR_NO_MORE_ITEMS { 
                    $verboseDetails = "No devices match or there is already a better driver"
                    $deviceStatus = 'Successful'  
                }
                $pnpUtilErrorCode_ERROR_SUCCESS_REBOOT_INITIATED {
                    $verboseDetails = "A system reboot is underway"
                    $deviceStatus = 'Successful'
                    $returnCode = $returnCode_REBOOT_REQUIRED
                    $rebootRequired = $true
                }
                $pnpUtilErrorCode_ERROR_SUCCESS_REBOOT_REQUIRED {
                    $verboseDetails = "A system reboot is required"
                    $deviceStatus = 'Successful'  
                    $returnCode = $returnCode_REBOOT_REQUIRED
                    $rebootRequired = $true
                }
            }
        }
        else {
            $deviceStatus = 'Successful'
        }
        if (-not $Verbose) {
            $verboseDetails = ""
        }
        if ($deviceStatus -ne 'Successful') {
            $componentsFailed += 1
        }
        if ($Verbose) {
            $actionResult.AddTableRow(
                [PSCustomObject]@{Component = $component.DisplayName; Status = $deviceStatus; ErrorCode = $errorCode; 
                    Details = $verboseDetails; PnpUtilOutput = $pnpUtilOutput 
                }
            )
        }
        else {
            $actionResult.AddTableRow(
                [PSCustomObject]@{Component = $component.DisplayName; Status = $deviceStatus; ErrorCode = $errorCode;
                    Details = $verboseDetails 
                }
            )
        }
        
    }
    if ($componentsFailed -gt 0) {
        $actionResult.ReturnCode = $returnCode_INSTALLATION_FAILED
        return $actionResult
    }
    if ($componentsFound -eq 0) {
        $actionResult.ReturnCode = $returnCode_NO_INF_FILES
        return $actionResult
    }
    if ($rebootRequired) {
        $actionResult.ReturnCode = $returnCode_REBOOT_REQUIRED
    }
    else {
        $actionResult.ReturnCode = $returnCode_SUCCESS
    }
    return $actionResult
}

function Get-PartialNameMatch {

    $devices = Get-PnpDevice -ErrorAction Stop
    $partialNameMatchList = @($SOL_PATTERN)
    $matchesFound = 0
    
    foreach ($device in $devices) {
        foreach ($pattern in $partialNameMatchList) {
            if ($device.FriendlyName -like "$pattern*") {
                $matchesFound += 1
                foreach ($component in $components) {
                    if ($component.SearchName -eq $pattern) {
                        $component.SearchName = $device.FriendlyName
                        break
                    }
                }
                if ($matchesFound -eq $partialNameMatchList.Count) {
                    return
                }
                break
            }
        }
    }
}

function Update-Components-Information {
    Get-PartialNameMatch
    Get-ExtensionsInformation
}

function Uninstall-Software {
    $actionResult = [ActionResult]::new()
    $actionResult.AddInformationBefore("[+] Starting uninstallation of CSE Software...")
    $actionResult.ReturnCode = $returnCode_SUCCESS
    $rebootRequired = $false

    [Array]::Reverse($components)
	
    Update-Components-Information

    $componentsFailed = 0
    foreach ( $component in $components ) {
        $componentFailed = $true
        $errorCode = ""
        $verboseDetails = ""
        $oemFilename = ""
        try {
            if ($component.OEMInfName -eq "") {
                $pnpDevices = Get-PnpDevice -ErrorAction Stop -FriendlyName $component.SearchName | Where-Object { $_.Present -eq $true }
                if ($pnpDevices -is [Array]) {
                    $actionResult.AddTableRow(
                        [PSCustomObject]@{Component = $component.DisplayName; Status = "Multiple devices"; ErrorCode = "";
                            Details = "Manual uninstall required"; PnpUtilOutput = ""
                        }
                    )
                    $componentsFailed += 1
                    continue
                }
                if ($null -eq $pnpDevices) {
                    $actionResult.AddTableRow(
                        [PSCustomObject]@{Component = $component.DisplayName; Status = 'Not installed'; ErrorCode = ""; 
                            Details = ""; PnpUtilOutput = ""  
                        }
                    )
                    continue
                }
                $pnpDevice = $pnpDevices
                $oemFilename = (Get-PnpDeviceProperty -InputObject $pnpDevice -KeyName 'DEVPKEY_Device_DriverInfPath').Data
            }
            else {
                $oemFilename = $component.OEMInfName
            }

            $pnpUtilOutput = & $pnpUtil /delete-driver $oemFilename /uninstall /force
            $pnpUtilOutput = "Inf:" + $component.InfName + " " + $pnpUtilOutput
            $errorCode = $LASTEXITCODE
            if ((-Not $?) -or ($errorCode -ne 0)) {
                $deviceStatus = 'Failed'
                if ($errorCode -eq $pnpUtilErrorCode_ERROR_DRIVER_MATCHING_INF_NOT_FOUND) {
                    $verboseDetails = "Driver matching to INF not found."
                }
                if ($errorCode -eq $pnpUtilErrorCode_ERROR_FILE_NOT_INSTALLED_INF) {
                    $verboseDetails = "The specified file is not an installed OEM INF."
                }
                if (($errorCode -eq $pnpUtilErrorCode_ERROR_SUCCESS_REBOOT_REQUIRED) -or
                    ($errorCode -eq $pnpUtilErrorCode_ERROR_SUCCESS_REBOOT_INITIATED) ) {
                    $componentFailed = $false
                    $deviceStatus = 'Successful'
                    $verboseDetails = "Reboot required."
                    $rebootRequired = $true
                }
            }
            else {
                $componentFailed = $false
                $deviceStatus = 'Successful'
            }
        } 
        catch [Microsoft.PowerShell.Cmdletization.Cim.CimJobException] {
            $componentFailed = $false
            $verboseDetails = ""
            $deviceStatus = 'Not installed'
        }
        catch {
            $verboseDetails = $PSItem.Exception
            $deviceStatus = 'Error'
        }
        if (-not $Verbose) {
            $verboseDetails = ""
        }
        if ($componentFailed) {
            $componentsFailed += 1
        }
        if ($Verbose) {
            $actionResult.AddTableRow(
                [PSCustomObject]@{Component = $component.DisplayName; Status = $deviceStatus; ErrorCode = $errorCode; 
                    Details = $verboseDetails; PnpUtilOutput = $pnpUtilOutput 
                }
            )
        }
        else {
            $actionResult.AddTableRow(
                [PSCustomObject]@{Component = $component.DisplayName; Status = $deviceStatus; ErrorCode = $errorCode;
                    Details = $verboseDetails 
                }
            )
        }
    }

    if ($componentsFailed -gt 0) {
        $actionResult.ReturnCode = $returnCode_UNINSTALLATION_FAILED
        return $actionResult
    }
    if ($rebootRequired) {
        $actionResult.ReturnCode = $returnCode_REBOOT_REQUIRED
    }
    else {
        $actionResult.ReturnCode = $returnCode_SUCCESS
    }
    return $actionResult
}

# Main script logic
function Main {
    switch ($action) {
        $action_INFORMATION {
            return Show-Information
        }
        $action_INSTALL {
            return Install-Software
        }
        $action_UNINSTALL {
            return Uninstall-Software
        }
    }
}

Show-Header
$res = Main

# We cannot encapsulate print logic in the class ActionResult
# For details: https://learn.microsoft.com/en-us/powershell/module/microsoft.powershell.core/about/about_classes?view=powershell-6#output-in-class-methods
foreach ($info in $res.InformationBefore) {
    PrintString $info
}
if (-not $Silent) {
    $res.Table | Format-Table -Wrap -AutoSize
}
foreach ($info in $res.InformationAfter) {
    PrintString $info
}
Exit-Script $res.ReturnCode

# SIG # Begin signature block
# MIItoQYJKoZIhvcNAQcCoIItkjCCLY4CAQExDzANBglghkgBZQMEAgIFADCBiQYK
# KwYBBAGCNwIBBKB7MHkwNAYKKwYBBAGCNwIBHjAmAgMBAAAEEB/MO2BZSwhOtyTS
# xil+81ECAQACAQACAQACAQACAQAwQTANBglghkgBZQMEAgIFAAQwviNC11gov6Is
# bbvf+X40d8jQhcMKLH+3r23zI738D1KIspjgYUr64X/YYGvHAL2KoIIRWDCCBW8w
# ggRXoAMCAQICEEj8k7RgVZSNNqfJionWlBYwDQYJKoZIhvcNAQEMBQAwezELMAkG
# A1UEBhMCR0IxGzAZBgNVBAgMEkdyZWF0ZXIgTWFuY2hlc3RlcjEQMA4GA1UEBwwH
# U2FsZm9yZDEaMBgGA1UECgwRQ29tb2RvIENBIExpbWl0ZWQxITAfBgNVBAMMGEFB
# QSBDZXJ0aWZpY2F0ZSBTZXJ2aWNlczAeFw0yMTA1MjUwMDAwMDBaFw0yODEyMzEy
# MzU5NTlaMFYxCzAJBgNVBAYTAkdCMRgwFgYDVQQKEw9TZWN0aWdvIExpbWl0ZWQx
# LTArBgNVBAMTJFNlY3RpZ28gUHVibGljIENvZGUgU2lnbmluZyBSb290IFI0NjCC
# AiIwDQYJKoZIhvcNAQEBBQADggIPADCCAgoCggIBAI3nlBIiBCR0Lv8WIwKSirau
# NoWsR9QjkSs+3H3iMaBRb6yEkeNSirXilt7Qh2MkiYr/7xKTO327toq9vQV/J5tr
# ZdOlDGmxvEk5mvFtbqrkoIMn2poNK1DpS1uzuGQ2pH5KPalxq2Gzc7M8Cwzv2zNX
# 5b40N+OXG139HxI9ggN25vs/ZtKUMWn6bbM0rMF6eNySUPJkx6otBKvDaurgL6en
# 3G7X6P/aIatAv7nuDZ7G2Z6Z78beH6kMdrMnIKHWuv2A5wHS7+uCKZVwjf+7Fc/+
# 0Q82oi5PMpB0RmtHNRN3BTNPYy64LeG/ZacEaxjYcfrMCPJtiZkQsa3bPizkqhiw
# xgcBdWfebeljYx42f2mJvqpFPm5aX4+hW8udMIYw6AOzQMYNDzjNZ6hTiPq4MGX6
# b8fnHbGDdGk+rMRoO7HmZzOatgjggAVIQO72gmRGqPVzsAaV8mxln79VWxycVxrH
# eEZ8cKqUG4IXrIfptskOgRxA1hYXKfxcnBgr6kX1773VZ08oXgXukEx658b00Pz6
# zT4yRhMgNooE6reqB0acDZM6CWaZWFwpo7kMpjA4PNBGNjV8nLruw9X5Cnb6fgUb
# QMqSNenVetG1fwCuqZCqxX8BnBCxFvzMbhjcb2L+plCnuHu4nRU//iAMdcgiWhOV
# GZAA6RrVwobx447sX/TlAgMBAAGjggESMIIBDjAfBgNVHSMEGDAWgBSgEQojPpbx
# B+zirynvgqV/0DCktDAdBgNVHQ4EFgQUMuuSmv81lkgvKEBCcCA2kVwXheYwDgYD
# VR0PAQH/BAQDAgGGMA8GA1UdEwEB/wQFMAMBAf8wEwYDVR0lBAwwCgYIKwYBBQUH
# AwMwGwYDVR0gBBQwEjAGBgRVHSAAMAgGBmeBDAEEATBDBgNVHR8EPDA6MDigNqA0
# hjJodHRwOi8vY3JsLmNvbW9kb2NhLmNvbS9BQUFDZXJ0aWZpY2F0ZVNlcnZpY2Vz
# LmNybDA0BggrBgEFBQcBAQQoMCYwJAYIKwYBBQUHMAGGGGh0dHA6Ly9vY3NwLmNv
# bW9kb2NhLmNvbTANBgkqhkiG9w0BAQwFAAOCAQEAEr+h74t0mphEuGlGtaskCgyk
# ime4OoG/RYp9UgeojR9OIYU5o2teLSCGvxC4rnk7U820+9hEvgbZXGNn1EAWh0SG
# cirWMhX1EoPC+eFdEUBn9kIncsUj4gI4Gkwg4tsB981GTyaifGbAUTa2iQJUx/xY
# +2wA7v6Ypi6VoQxTKR9v2BmmT573rAnqXYLGi6+Ap72BSFKEMdoy7BXkpkw9bDlz
# 1AuFOSDghRpo4adIOKnRNiV3wY0ZFsWITGZ9L2POmOhp36w8qF2dyRxbrtjzL3TP
# uH7214OdEZZimq5FE9p/3Ef738NSn+YGVemdjPI6YlG87CQPKdRYgITkRXta2DCC
# BcMwggQroAMCAQICEHqVHgg2WvB2BEAdFLTzupEwDQYJKoZIhvcNAQEMBQAwVDEL
# MAkGA1UEBhMCR0IxGDAWBgNVBAoTD1NlY3RpZ28gTGltaXRlZDErMCkGA1UEAxMi
# U2VjdGlnbyBQdWJsaWMgQ29kZSBTaWduaW5nIENBIFIzNjAeFw0yNTAyMTkwMDAw
# MDBaFw0yNjAyMTkyMzU5NTlaMFoxCzAJBgNVBAYTAlVTMRMwEQYDVQQIDApDYWxp
# Zm9ybmlhMRowGAYDVQQKDBFJbnRlbCBDb3Jwb3JhdGlvbjEaMBgGA1UEAwwRSW50
# ZWwgQ29ycG9yYXRpb24wggGiMA0GCSqGSIb3DQEBAQUAA4IBjwAwggGKAoIBgQC7
# ctWHp2WukDL/iZJ5Evpu+wnlTpOj4mnnm446XRUIiE0T5EGA4bwHTl6qGE4fhqmT
# u6ekyRjPk/gOs3fGQ3XjruB9c6zm9RyCFe6YAfkW0ui2PHo+PfStZptuNqeI3Ur8
# qj53KtZ0uGwNtiga7cimlI4EevMKnJ0dUMry+JUPlbbdxnX9Xj+CKhxm2W0CfuI/
# RfW/zH0gMX6tLuogeN0ix5k+lZs/NPxRGIEOUkqC3bgPoQ7hPPJVsYWldc3RQ63C
# CVKLgUkYaGnTXd3MNbIjy+o6oVbih3k3sUqhFg1UfVxMLtHjA2nLzPyOK7fW/l6P
# gKGURJnFaii48HXfwJHwI3FWydn8YSxUYDpvKWAWplcxNnOEuDaIcxBczHVdPQGS
# AIwoJBpvvIAvbTQ3Z8XAtIrd0YL9VYgCUOeAqSXOcgs9CIj9nnrrmUEKxeEid+1W
# 7Z8pKWK8K+8Yke0G3ixfCSXuyVweJWlaJvW/mdys+xvOWMUDSl7EMndMdoDUMPMC
# AwEAAaOCAYkwggGFMB8GA1UdIwQYMBaAFA8qyyCHKLjsb0iuK1SmKaoXpM0MMB0G
# A1UdDgQWBBSO3zfJ+ielYF2FeRwEGUjzxX2aCDAOBgNVHQ8BAf8EBAMCB4AwDAYD
# VR0TAQH/BAIwADATBgNVHSUEDDAKBggrBgEFBQcDAzBKBgNVHSAEQzBBMDUGDCsG
# AQQBsjEBAgEDAjAlMCMGCCsGAQUFBwIBFhdodHRwczovL3NlY3RpZ28uY29tL0NQ
# UzAIBgZngQwBBAEwSQYDVR0fBEIwQDA+oDygOoY4aHR0cDovL2NybC5zZWN0aWdv
# LmNvbS9TZWN0aWdvUHVibGljQ29kZVNpZ25pbmdDQVIzNi5jcmwweQYIKwYBBQUH
# AQEEbTBrMEQGCCsGAQUFBzAChjhodHRwOi8vY3J0LnNlY3RpZ28uY29tL1NlY3Rp
# Z29QdWJsaWNDb2RlU2lnbmluZ0NBUjM2LmNydDAjBggrBgEFBQcwAYYXaHR0cDov
# L29jc3Auc2VjdGlnby5jb20wDQYJKoZIhvcNAQEMBQADggGBAFy6xht7KeNCUeMs
# SSkyj0/x7XR9XdEGJnHKF645UBAP/grRzwp2stFv9JjY/ikQ5vqu1k16JWqXFOk9
# oUX9Ik/Csm2KxtetybzT2XiLMhq3HUL26fm1RPKDcK6ryBdSButZPeSIK4xFWZQW
# aruPokORS9AuXy6RndCoacy1uNI5ZzONt4GEqng2LMQohdjEDauCUluK+sQ3t09Y
# f9Mzuux6D/ldXAzsW8XoR6mqQmZ4mX2rNPAKBNyQIZ/cAsmkH+AaElhMdQE0bYtR
# dKKIBUuC1UNZrDjGbpHuymvkFd7zszZy0flFkyUj2bBsgwMkIO5Gh84QT+HRd/qM
# pU6+EwVjphG7XLkry6EnQJxw3ACxPsK1CQUoN0h81ncJyH8KM5TKQxR4DPBVwKKh
# edXn6+FMdMjrJBljlwTfLDHbHE/cuq5zSTuaixxK1Z+lvX33kF4gr2G5cBOxGoCt
# jkoElOqXec9ptBLRPCKOBs7nEMMNeO3u+8+eTdnThBWrYw+BKTCCBhowggQCoAMC
# AQICEGIdbQxSAZ47kHkVIIkhHAowDQYJKoZIhvcNAQEMBQAwVjELMAkGA1UEBhMC
# R0IxGDAWBgNVBAoTD1NlY3RpZ28gTGltaXRlZDEtMCsGA1UEAxMkU2VjdGlnbyBQ
# dWJsaWMgQ29kZSBTaWduaW5nIFJvb3QgUjQ2MB4XDTIxMDMyMjAwMDAwMFoXDTM2
# MDMyMTIzNTk1OVowVDELMAkGA1UEBhMCR0IxGDAWBgNVBAoTD1NlY3RpZ28gTGlt
# aXRlZDErMCkGA1UEAxMiU2VjdGlnbyBQdWJsaWMgQ29kZSBTaWduaW5nIENBIFIz
# NjCCAaIwDQYJKoZIhvcNAQEBBQADggGPADCCAYoCggGBAJsrnVP6NT+OYAZDasDP
# 9X/2yFNTGMjO02x+/FgHlRd5ZTMLER4ARkZsQ3hAyAKwktlQqFZOGP/I+rLSJJmF
# eRno+DYDY1UOAWKA4xjMHY4qF2p9YZWhhbeFpPb09JNqFiTCYy/Rv/zedt4QJuIx
# eFI61tqb7/foXT1/LW2wHyN79FXSYiTxcv+18Irpw+5gcTbXnDOsrSHVJYdPE9s+
# 5iRF2Q/TlnCZGZOcA7n9qudjzeN43OE/TpKF2dGq1mVXn37zK/4oiETkgsyqA5lg
# AQ0c1f1IkOb6rGnhWqkHcxX+HnfKXjVodTmmV52L2UIFsf0l4iQ0UgKJUc2RGarh
# OnG3B++OxR53LPys3J9AnL9o6zlviz5pzsgfrQH4lrtNUz4Qq/Va5MbBwuahTcWk
# 4UxuY+PynPjgw9nV/35gRAhC3L81B3/bIaBb659+Vxn9kT2jUztrkmep/aLb+4xJ
# bKZHyvahAEx2XKHafkeKtjiMqcUf/2BG935A591GsllvWwIDAQABo4IBZDCCAWAw
# HwYDVR0jBBgwFoAUMuuSmv81lkgvKEBCcCA2kVwXheYwHQYDVR0OBBYEFA8qyyCH
# KLjsb0iuK1SmKaoXpM0MMA4GA1UdDwEB/wQEAwIBhjASBgNVHRMBAf8ECDAGAQH/
# AgEAMBMGA1UdJQQMMAoGCCsGAQUFBwMDMBsGA1UdIAQUMBIwBgYEVR0gADAIBgZn
# gQwBBAEwSwYDVR0fBEQwQjBAoD6gPIY6aHR0cDovL2NybC5zZWN0aWdvLmNvbS9T
# ZWN0aWdvUHVibGljQ29kZVNpZ25pbmdSb290UjQ2LmNybDB7BggrBgEFBQcBAQRv
# MG0wRgYIKwYBBQUHMAKGOmh0dHA6Ly9jcnQuc2VjdGlnby5jb20vU2VjdGlnb1B1
# YmxpY0NvZGVTaWduaW5nUm9vdFI0Ni5wN2MwIwYIKwYBBQUHMAGGF2h0dHA6Ly9v
# Y3NwLnNlY3RpZ28uY29tMA0GCSqGSIb3DQEBDAUAA4ICAQAG/4Lhd2M2bnuhFSCb
# E/8E/ph1RGHDVpVx0ZE/haHrQECxyNbgcv2FymQ5PPmNS6Dah66dtgCjBsULYAor
# 5wxxcgEPRl05pZOzI3IEGwwsepp+8iGsLKaVpL3z5CmgELIqmk/Q5zFgR1TSGmxq
# oEEhk60FqONzDn7D8p4W89h8sX+V1imaUb693TGqWp3T32IKGfIgy9jkd7GM7YCa
# 2xulWfQ6E1xZtYNEX/ewGnp9ZeHPsNwwviJMBZL4xVd40uPWUnOJUoSiugaz0yWL
# ODRtQxs5qU6E58KKmfHwJotl5WZ7nIQuDT0mWjwEx7zSM7fs9Tx6N+Q/3+49qTtU
# vAQsrEAxwmzOTJ6Jp6uWmHCgrHW4dHM3ITpvG5Ipy62KyqYovk5O6cC+040Si15K
# JpuQ9VJnbPvqYqfMB9nEKX/d2rd1Q3DiuDexMKCCQdJGpOqUsxLuCOuFOoGbO7Uv
# 3RjUpY39jkkp0a+yls6tN85fJe+Y8voTnbPU1knpy24wUFBkfenBa+pRFHwCBB1Q
# tS+vGNRhsceP3kSPNrrfN2sRzFYsNfrFaWz8YOdU254qNZQfd9O/VjxZ2Gjr3xgA
# NHtM3HxfzPYF6/pKK8EE4dj66qKKtm2DTL1KFCg/OYJyfrdLJq1q2/HXntgr2GVw
# +ZWhrWgMTn8v1SjZsLlrgIfZHDGCG44wghuKAgEBMGgwVDELMAkGA1UEBhMCR0Ix
# GDAWBgNVBAoTD1NlY3RpZ28gTGltaXRlZDErMCkGA1UEAxMiU2VjdGlnbyBQdWJs
# aWMgQ29kZSBTaWduaW5nIENBIFIzNgIQepUeCDZa8HYEQB0UtPO6kTANBglghkgB
# ZQMEAgIFAKCBjDAQBgorBgEEAYI3AgEMMQIwADAZBgkqhkiG9w0BCQMxDAYKKwYB
# BAGCNwIBBDAcBgorBgEEAYI3AgELMQ4wDAYKKwYBBAGCNwIBFTA/BgkqhkiG9w0B
# CQQxMgQwsxN2ij4n0LcNUAEpF2B0LKfVUOA4D9ajJeNd+OXe1hXxsXfRZlRlVgaY
# Eq7AG6DbMA0GCSqGSIb3DQEBAQUABIIBgD6WruLH6vAfk9xrAW05X4kpfDGMvbok
# MuOhFDJzAN5PSOBAFpCil/6ghVudz5ZuuympX2FElR5phFmLkC0X3cqX0rQ0waf0
# y7qZRQJ+d/HlC2/F9RgH4hHRQF9ebA63p4j3cLCpA9VXzUePOgo53+D7tcGBZxx0
# BL5s+2yme8kxaErkoOxY0WHGN/RHgVBPWUYm93ks8J4RM5mo6rl2+utUkPck+fpJ
# V/qnYPxJxwgFEYrE5EnsvnaMsX/VD4KtZSI8cEmE6QSvnqdu09X2hM90U5H/0Uwh
# tgCHA8cE5zQD2zdavgGo1ZvwNTtXeoHDvEFr2s6f5xr8r/6sqPWrQFQs+13qI3Co
# ncUOsW6T062r0/JO4HPluDnoWzqgIdNcb32dMb3yS5t+yfQa+oH93RZqFxbT2gu6
# 2KVKlmOc12pkptqH5pElRDQIQolCVNYQMiFfozMQJdUghSa74i+y25rrJY1elUCM
# xlSRoewoSuhw0+0g694zUIWA3cUWZt0wi6GCGOgwghjkBgorBgEEAYI3AwMBMYIY
# 1DCCGNAGCSqGSIb3DQEHAqCCGMEwghi9AgEDMQ8wDQYJYIZIAWUDBAICBQAwggEH
# BgsqhkiG9w0BCRABBKCB9wSB9DCB8QIBAQYKKwYBBAGyMQIBATBBMA0GCWCGSAFl
# AwQCAgUABDCQoYM0lOe2A8KLaisS+7DkneUP2kU++0V12CUVpJ3+oVfb7x1yOUwW
# zD3EMHGlBKoCFFHVXgEtN9iRX7IO1YXi5q1OSQy0GA8yMDI1MDkxNTEyNTQwNFqg
# dqR0MHIxCzAJBgNVBAYTAkdCMRcwFQYDVQQIEw5XZXN0IFlvcmtzaGlyZTEYMBYG
# A1UEChMPU2VjdGlnbyBMaW1pdGVkMTAwLgYDVQQDEydTZWN0aWdvIFB1YmxpYyBU
# aW1lIFN0YW1waW5nIFNpZ25lciBSMzagghMEMIIGYjCCBMqgAwIBAgIRAKQpO24e
# 3denNAiHrXpOtyQwDQYJKoZIhvcNAQEMBQAwVTELMAkGA1UEBhMCR0IxGDAWBgNV
# BAoTD1NlY3RpZ28gTGltaXRlZDEsMCoGA1UEAxMjU2VjdGlnbyBQdWJsaWMgVGlt
# ZSBTdGFtcGluZyBDQSBSMzYwHhcNMjUwMzI3MDAwMDAwWhcNMzYwMzIxMjM1OTU5
# WjByMQswCQYDVQQGEwJHQjEXMBUGA1UECBMOV2VzdCBZb3Jrc2hpcmUxGDAWBgNV
# BAoTD1NlY3RpZ28gTGltaXRlZDEwMC4GA1UEAxMnU2VjdGlnbyBQdWJsaWMgVGlt
# ZSBTdGFtcGluZyBTaWduZXIgUjM2MIICIjANBgkqhkiG9w0BAQEFAAOCAg8AMIIC
# CgKCAgEA04SV9G6kU3jyPRBLeBIHPNyUgVNnYayfsGOyYEXrn3+SkDYTLs1crcw/
# ol2swE1TzB2aR/5JIjKNf75QBha2Ddj+4NEPKDxHEd4dEn7RTWMcTIfm492TW22I
# 8LfH+A7Ehz0/safc6BbsNBzjHTt7FngNfhfJoYOrkugSaT8F0IzUh6VUwoHdYDpi
# ln9dh0n0m545d5A5tJD92iFAIbKHQWGbCQNYplqpAFasHBn77OqW37P9BhOASdmj
# p3IijYiFdcA0WQIe60vzvrk0HG+iVcwVZjz+t5OcXGTcxqOAzk1frDNZ1aw8nFhG
# EvG0ktJQknnJZE3D40GofV7O8WzgaAnZmoUn4PCpvH36vD4XaAF2CjiPsJWiY/j2
# xLsJuqx3JtuI4akH0MmGzlBUylhXvdNVXcjAuIEcEQKtOBR9lU4wXQpISrbOT8ux
# +96GzBq8TdbhoFcmYaOBZKlwPP7pOp5Mzx/UMhyBA93PQhiCdPfIVOCINsUY4U23
# p4KJ3F1HqP3H6Slw3lHACnLilGETXRg5X/Fp8G8qlG5Y+M49ZEGUp2bneRLZoyHT
# yynHvFISpefhBCV0KdRZHPcuSL5OAGWnBjAlRtHvsMBrI3AAA0Tu1oGvPa/4yeei
# Ayu+9y3SLC98gDVbySnXnkujjhIh+oaatsk/oyf5R2vcxHahajMCAwEAAaOCAY4w
# ggGKMB8GA1UdIwQYMBaAFF9Y7UwxeqJhQo1SgLqzYZcZojKbMB0GA1UdDgQWBBSI
# YYyhKjdkgShgoZsx0Iz9LALOTzAOBgNVHQ8BAf8EBAMCBsAwDAYDVR0TAQH/BAIw
# ADAWBgNVHSUBAf8EDDAKBggrBgEFBQcDCDBKBgNVHSAEQzBBMDUGDCsGAQQBsjEB
# AgEDCDAlMCMGCCsGAQUFBwIBFhdodHRwczovL3NlY3RpZ28uY29tL0NQUzAIBgZn
# gQwBBAIwSgYDVR0fBEMwQTA/oD2gO4Y5aHR0cDovL2NybC5zZWN0aWdvLmNvbS9T
# ZWN0aWdvUHVibGljVGltZVN0YW1waW5nQ0FSMzYuY3JsMHoGCCsGAQUFBwEBBG4w
# bDBFBggrBgEFBQcwAoY5aHR0cDovL2NydC5zZWN0aWdvLmNvbS9TZWN0aWdvUHVi
# bGljVGltZVN0YW1waW5nQ0FSMzYuY3J0MCMGCCsGAQUFBzABhhdodHRwOi8vb2Nz
# cC5zZWN0aWdvLmNvbTANBgkqhkiG9w0BAQwFAAOCAYEAAoE+pIZyUSH5ZakuPVKK
# 4eWbzEsTRJOEjbIu6r7vmzXXLpJx4FyGmcqnFZoa1dzx3JrUCrdG5b//LfAxOGy9
# Ph9JtrYChJaVHrusDh9NgYwiGDOhyyJ2zRy3+kdqhwtUlLCdNjFjakTSE+hkC9F5
# ty1uxOoQ2ZkfI5WM4WXA3ZHcNHB4V42zi7Jk3ktEnkSdViVxM6rduXW0jmmiu71Z
# pBFZDh7Kdens+PQXPgMqvzodgQJEkxaION5XRCoBxAwWwiMm2thPDuZTzWp/gUFz
# i7izCmEt4pE3Kf0MOt3ccgwn4Kl2FIcQaV55nkjv1gODcHcD9+ZVjYZoyKTVWb4V
# qMQy/j8Q3aaYd/jOQ66Fhk3NWbg2tYl5jhQCuIsE55Vg4N0DUbEWvXJxtxQQaVR5
# xzhEI+BjJKzh3TQ026JxHhr2fuJ0mV68AluFr9qshgwS5SpN5FFtaSEnAwqZv3IS
# +mlG50rK7W3qXbWwi4hmpylUfygtYLEdLQukNEX1jiOKMIIGFDCCA/ygAwIBAgIQ
# eiOu2lNplg+RyD5c9MfjPzANBgkqhkiG9w0BAQwFADBXMQswCQYDVQQGEwJHQjEY
# MBYGA1UEChMPU2VjdGlnbyBMaW1pdGVkMS4wLAYDVQQDEyVTZWN0aWdvIFB1Ymxp
# YyBUaW1lIFN0YW1waW5nIFJvb3QgUjQ2MB4XDTIxMDMyMjAwMDAwMFoXDTM2MDMy
# MTIzNTk1OVowVTELMAkGA1UEBhMCR0IxGDAWBgNVBAoTD1NlY3RpZ28gTGltaXRl
# ZDEsMCoGA1UEAxMjU2VjdGlnbyBQdWJsaWMgVGltZSBTdGFtcGluZyBDQSBSMzYw
# ggGiMA0GCSqGSIb3DQEBAQUAA4IBjwAwggGKAoIBgQDNmNhDQatugivs9jN+JjTk
# iYzT7yISgFQ+7yavjA6Bg+OiIjPm/N/t3nC7wYUrUlY3mFyI32t2o6Ft3EtxJXCc
# 5MmZQZ8AxCbh5c6WzeJDB9qkQVa46xiYEpc81KnBkAWgsaXnLURoYZzksHIzzCNx
# tIXnb9njZholGw9djnjkTdAA83abEOHQ4ujOGIaBhPXG2NdV8TNgFWZ9BojlAvfl
# xNMCOwkCnzlH4oCw5+4v1nssWeN1y4+RlaOywwRMUi54fr2vFsU5QPrgb6tSjvEU
# h1EC4M29YGy/SIYM8ZpHadmVjbi3Pl8hJiTWw9jiCKv31pcAaeijS9fc6R7DgyyL
# IGflmdQMwrNRxCulVq8ZpysiSYNi79tw5RHWZUEhnRfs/hsp/fwkXsynu1jcsUX+
# HuG8FLa2BNheUPtOcgw+vHJcJ8HnJCrcUWhdFczf8O+pDiyGhVYX+bDDP3GhGS7T
# mKmGnbZ9N+MpEhWmbiAVPbgkqykSkzyYVr15OApZYK8CAwEAAaOCAVwwggFYMB8G
# A1UdIwQYMBaAFPZ3at0//QET/xahbIICL9AKPRQlMB0GA1UdDgQWBBRfWO1MMXqi
# YUKNUoC6s2GXGaIymzAOBgNVHQ8BAf8EBAMCAYYwEgYDVR0TAQH/BAgwBgEB/wIB
# ADATBgNVHSUEDDAKBggrBgEFBQcDCDARBgNVHSAECjAIMAYGBFUdIAAwTAYDVR0f
# BEUwQzBBoD+gPYY7aHR0cDovL2NybC5zZWN0aWdvLmNvbS9TZWN0aWdvUHVibGlj
# VGltZVN0YW1waW5nUm9vdFI0Ni5jcmwwfAYIKwYBBQUHAQEEcDBuMEcGCCsGAQUF
# BzAChjtodHRwOi8vY3J0LnNlY3RpZ28uY29tL1NlY3RpZ29QdWJsaWNUaW1lU3Rh
# bXBpbmdSb290UjQ2LnA3YzAjBggrBgEFBQcwAYYXaHR0cDovL29jc3Auc2VjdGln
# by5jb20wDQYJKoZIhvcNAQEMBQADggIBABLXeyCtDjVYDJ6BHSVY/UwtZ3Svx2Im
# IfZVVGnGoUaGdltoX4hDskBMZx5NY5L6SCcwDMZhHOmbyMhyOVJDwm1yrKYqGDHW
# zpwVkFJ+996jKKAXyIIaUf5JVKjccev3w16mNIUlNTkpJEor7edVJZiRJVCAmWAa
# Hcw9zP0hY3gj+fWp8MbOocI9Zn78xvm9XKGBp6rEs9sEiq/pwzvg2/KjXE2yWUQI
# kms6+yslCRqNXPjEnBnxuUB1fm6bPAV+Tsr/Qrd+mOCJemo06ldon4pJFbQd0TQV
# IMLv5koklInHvyaf6vATJP4DfPtKzSBPkKlOtyaFTAjD2Nu+di5hErEVVaMqSVbf
# Pzd6kNXOhYm23EWm6N2s2ZHCHVhlUgHaC4ACMRCgXjYfQEDtYEK54dUwPJXV7icz
# 0rgCzs9VI29DwsjVZFpO4ZIVR33LwXyPDbYFkLqYmgHjR3tKVkhh9qKV2WCmBuC2
# 7pIOx6TYvyqiYbntinmpOqh/QPAnhDgexKG9GX/n1PggkGi9HCapZp8fRwg8Rftw
# S21Ln61euBG0yONM6noD2XQPrFwpm3GcuqJMf0o8LLrFkSLRQNwxPDDkWXhW+gZs
# wbaiie5fd/W2ygcto78XCSPfFWveUOSZ5SqK95tBO8aTHmEa4lpJVD7HrTEn9jb1
# EGvxOb1cnn0CMIIGgjCCBGqgAwIBAgIQNsKwvXwbOuejs902y8l1aDANBgkqhkiG
# 9w0BAQwFADCBiDELMAkGA1UEBhMCVVMxEzARBgNVBAgTCk5ldyBKZXJzZXkxFDAS
# BgNVBAcTC0plcnNleSBDaXR5MR4wHAYDVQQKExVUaGUgVVNFUlRSVVNUIE5ldHdv
# cmsxLjAsBgNVBAMTJVVTRVJUcnVzdCBSU0EgQ2VydGlmaWNhdGlvbiBBdXRob3Jp
# dHkwHhcNMjEwMzIyMDAwMDAwWhcNMzgwMTE4MjM1OTU5WjBXMQswCQYDVQQGEwJH
# QjEYMBYGA1UEChMPU2VjdGlnbyBMaW1pdGVkMS4wLAYDVQQDEyVTZWN0aWdvIFB1
# YmxpYyBUaW1lIFN0YW1waW5nIFJvb3QgUjQ2MIICIjANBgkqhkiG9w0BAQEFAAOC
# Ag8AMIICCgKCAgEAiJ3YuUVnnR3d6LkmgZpUVMB8SQWbzFoVD9mUEES0QUCBdxSZ
# qdTkdizICFNeINCSJS+lV1ipnW5ihkQyC0cRLWXUJzodqpnMRs46npiJPHrfLBOi
# fjfhpdXJ2aHHsPHggGsCi7uE0awqKggE/LkYw3sqaBia67h/3awoqNvGqiFRJ+OT
# WYmUCO2GAXsePHi+/JUNAax3kpqstbl3vcTdOGhtKShvZIvjwulRH87rbukNyHGW
# X5tNK/WABKf+Gnoi4cmisS7oSimgHUI0Wn/4elNd40BFdSZ1EwpuddZ+Wr7+Dfo0
# lcHflm/FDDrOJ3rWqauUP8hsokDoI7D/yUVI9DAE/WK3Jl3C4LKwIpn1mNzMyptR
# wsXKrop06m7NUNHdlTDEMovXAIDGAvYynPt5lutv8lZeI5w3MOlCybAZDpK3Dy1M
# Ko+6aEtE9vtiTMzz/o2dYfdP0KWZwZIXbYsTIlg1YIetCpi5s14qiXOpRsKqFKqa
# v9R1R5vj3NgevsAsvxsAnI8Oa5s2oy25qhsoBIGo/zi6GpxFj+mOdh35Xn91y72J
# 4RGOJEoqzEIbW3q0b2iPuWLA911cRxgY5SJYubvjay3nSMbBPPFsyl6mY4/WYucm
# yS9lo3l7jk27MAe145GWxK4O3m3gEFEIkv7kRmefDR7Oe2T1HxAnICQvr9sCAwEA
# AaOCARYwggESMB8GA1UdIwQYMBaAFFN5v1qqK0rPVIDh2JvAnfKyA2bLMB0GA1Ud
# DgQWBBT2d2rdP/0BE/8WoWyCAi/QCj0UJTAOBgNVHQ8BAf8EBAMCAYYwDwYDVR0T
# AQH/BAUwAwEB/zATBgNVHSUEDDAKBggrBgEFBQcDCDARBgNVHSAECjAIMAYGBFUd
# IAAwUAYDVR0fBEkwRzBFoEOgQYY/aHR0cDovL2NybC51c2VydHJ1c3QuY29tL1VT
# RVJUcnVzdFJTQUNlcnRpZmljYXRpb25BdXRob3JpdHkuY3JsMDUGCCsGAQUFBwEB
# BCkwJzAlBggrBgEFBQcwAYYZaHR0cDovL29jc3AudXNlcnRydXN0LmNvbTANBgkq
# hkiG9w0BAQwFAAOCAgEADr5lQe1oRLjlocXUEYfktzsljOt+2sgXke3Y8UPEooU5
# y39rAARaAdAxUeiX1ktLJ3+lgxtoLQhn5cFb3GF2SSZRX8ptQ6IvuD3wz/LNHKpQ
# 5nX8hjsDLRhsyeIiJsms9yAWnvdYOdEMq1W61KE9JlBkB20XBee6JaXx4UBErc+Y
# uoSb1SxVf7nkNtUjPfcxuFtrQdRMRi/fInV/AobE8Gw/8yBMQKKaHt5eia8ybT8Y
# /Ffa6HAJyz9gvEOcF1VWXG8OMeM7Vy7Bs6mSIkYeYtddU1ux1dQLbEGur18ut97w
# gGwDiGinCwKPyFO7ApcmVJOtlw9FVJxw/mL1TbyBns4zOgkaXFnnfzg4qbSvnrwy
# j1NiurMp4pmAWjR+Pb/SIduPnmFzbSN/G8reZCL4fvGlvPFk4Uab/JVCSmj59+/m
# B2Gn6G/UYOy8k60mKcmaAZsEVkhOFuoj4we8CYyaR9vd9PGZKSinaZIkvVjbH/3n
# lLb0a7SBIkiRzfPfS9T+JesylbHa1LtRV9U/7m0q7Ma2CQ/t392ioOssXW7oKLdO
# mMBl14suVFBmbzrt5V5cQPnwtd3UOTpS9oCG+ZZheiIvPgkDmA8FzPsnfXW5qHEL
# B43ET7HHFHeRPRYrMBKjkb8/IN7Po0d0hQoF4TeMM+zYAJzoKQnVKOLg8pZVPT8x
# ggSSMIIEjgIBATBqMFUxCzAJBgNVBAYTAkdCMRgwFgYDVQQKEw9TZWN0aWdvIExp
# bWl0ZWQxLDAqBgNVBAMTI1NlY3RpZ28gUHVibGljIFRpbWUgU3RhbXBpbmcgQ0Eg
# UjM2AhEApCk7bh7d16c0CIetek63JDANBglghkgBZQMEAgIFAKCCAfkwGgYJKoZI
# hvcNAQkDMQ0GCyqGSIb3DQEJEAEEMBwGCSqGSIb3DQEJBTEPFw0yNTA5MTUxMjU0
# MDRaMD8GCSqGSIb3DQEJBDEyBDDyWngrDcJDuxFFS7pAXqo0bTWL1pzOUAJ4TEsG
# 66RUcSDg/vy8ZyE97ALopmEoKkkwggF6BgsqhkiG9w0BCRACDDGCAWkwggFlMIIB
# YTAWBBQ4yRSBEES03GY+k9R0S4FBhqm1sTCBhwQUxq5U5HiG8Xw9VRJIjGnDSnr5
# wt0wbzBbpFkwVzELMAkGA1UEBhMCR0IxGDAWBgNVBAoTD1NlY3RpZ28gTGltaXRl
# ZDEuMCwGA1UEAxMlU2VjdGlnbyBQdWJsaWMgVGltZSBTdGFtcGluZyBSb290IFI0
# NgIQeiOu2lNplg+RyD5c9MfjPzCBvAQUhT1jLZOCgmF80JA1xJHeksFC2scwgaMw
# gY6kgYswgYgxCzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpOZXcgSmVyc2V5MRQwEgYD
# VQQHEwtKZXJzZXkgQ2l0eTEeMBwGA1UEChMVVGhlIFVTRVJUUlVTVCBOZXR3b3Jr
# MS4wLAYDVQQDEyVVU0VSVHJ1c3QgUlNBIENlcnRpZmljYXRpb24gQXV0aG9yaXR5
# AhA2wrC9fBs656Oz3TbLyXVoMA0GCSqGSIb3DQEBAQUABIICAKf7eTOxKvA1eGgd
# URGT3eFTOlSY+NSTx32c15IwYYtQfHiXJ8ZzE0BBHIKfM1FJi7YZb09HX6wXub4h
# DJEXj2gRJ2SdGRybJLGB/1TtAuacjX2rdH5Aw2Gz42MiflZ2ar2GCdwtncrvsQsu
# O81YBLQmsZ0Q7HJ2couMJ8lzsCP50t759SGWnLqY6zFfDvvBHm6U4kC3meZfb0SJ
# obLqWbeyT9jURjptlZPbYAj3ZJDsD0sWd/BwiejAbdJr9pa7zX2oiEvekhHT2UI8
# MQGDJyxs0qrEVV0SHpsqYcU/a+ClfVXgmInHH7HUhm5mYnxbMuvbqEsNQiTwX8S/
# DKDXVkMC4AwaR12c9vbv7PR1dvkWkqpcBJmV1YF6XVF9HVcT7Xzxp64ctBaP0GyW
# FD/t1b7UA2+GVU+3ytaprOzhyT9xsoWIFPrtrH1JKc8PDuUnDBvete23613m5V5Z
# 8BO7zEp+AdKDS0K5unWq5kg0SfOghKdNjG8f2KTivMLuFK8KA3ewTZl/1rTYbitQ
# 0hlmwQQvuwtp4SluPlEkuwXPXOoH6XUYZD5o6J4NlR2fYLc8ZT8qQw6J7YPkO/gb
# iAq41l7gPP8RPJ+xm5L08beNqwx+antkM8GGrGmSnT+yk0aDdKQXq2Lztnp1Hr/b
# iYZ65DhCy9QGaRHKGqOtpVy9pPZ8
# SIG # End signature block
