/*******************************************************************************
** INTEL CONFIDENTIAL
**
** Copyright (c) 2020-2023 Intel Corporation All Rights Reserved
**
** This software and the related documents are Intel copyrighted materials,
** and your use of them is governed by the express license under which they
** were provided to you ("License"). Unless the License provides otherwise,
** you may not use, modify, copy, publish, distribute, disclose or transmit
** this software or the related documents without Intel's prior written
** permission.
**
** The source code contained or described herein and all documents related to
** the source code ("Material") are owned by Intel Corporation or its suppliers
** or licensors.
**
** This software and the related documents are provided as is,
** with no express or implied warranties, other than those that are expressly
** stated in the License.
**
*******************************************************************************/

#include <iostream>
#include <memory>
#include <string>
#include <vector>

#include "Exceptions.h"
#include "JsonFields.h"
#include "ProviderDelegate.h"
#include "SampleBatteryProvider.h"

using Json = nlohmann::json;
using std::string;

// Mocked sample driver.
class MockedBatteryDriver
{
private:
	string x = "charging";

private:
	int batteryPercentage = 90;

public:
	string getState()
	{
		return x;
	}

public:
	void setState(string value)
	{
		x = value;
	}

public:
	void takeAction(Json& result)
	{
		result["Success"] = true;
		result["Error"] = "None";
	}

public:
	int getBatteryPercentage()
	{
		return batteryPercentage;
	}
};

static const std::string kDataSchema = R"(
{
    "$schema": "http://json-schema.org/draft-07/schema#",
    "type": "object",
    "properties": {
        "Manufacturer": {
            "type": "string",
            "readOnly": "false"
        },
        "Model": {
            "type": "string",
            "readOnly": "false",
            "description" : "Camera model name"
        },
        "State": {
            "type": "string",
            "description" : "state of the camera either charged or charging",
            "enum": ["charged", "charging"]
        },
        "IsIonBattery": {
            "type": "boolean",
            "readOnly": "true"
        },
        "ChildCell": {
            "type": "object",
            "properties": {
                "State": {
                    "type": "string",
                    "enum": ["charged", "charging"]
                }
            }
        }
    },
    "required": [
        "State"
    ]
}
)";

static const std::string kEventSchema = R"(
{
  "$schema": "http://json-schema.org/draft-07/schema",
  "type": "object",
  "properties": {
    "BatteryFullyCharged": {
      "$comment": "This is the event description raised by IPF framework.",
        "$id": "#/battery/batteryFullyCharged",
        "type": "object",
        "properties": {
          "State": {
            "type": "string",
            "enum": ["charged","charging"]
          }
        },
        "additionalProperties": false,
        "required": [ "State" ]
     },
    "BatteryAttach": {
      "$id": "#/root/battery/batteryAttach",
      "type": "object",
      "properties": {
        "InstanceName": {
          "type": "string"
        }
      },
      "additionalProperties": false,
      "required": [ "InstanceName" ]
    },
    "BatteryDetach": {
      "$id": "#/root/battery/BatteryDetach",
      "type": "object",
      "properties": {
        "InstanceName": {
          "type": "string"
        }
      },
      "additionalProperties": false,
      "required": [ "InstanceName" ]
    }
  },
  "additionalProperties": false
}
)";

static const std::string kCommandSchema = R"(
{
    "$schema": "http://json-schema.org/draft-07/schema",
    "type": "object",
    "properties": {
        "CommandName": {
            "type": "string"
        },
        "SimulateUnplug": {
          "$id": "#/root/battery/simulateUnplug",
          "type": "object",
          "properties": {
            "InstanceName": {
              "type": "string"
            }
          },
          "required": [ "InstanceName" ]
        },
        "SimulatePlug": {
          "$id": "#/root/battery/SimulatePlug",
          "type": "object",
          "properties": {
            "InstanceName": {
              "type": "string",
              "description" : "simulatePlug instance name"
            },
            "Manufacturer": {
              "type": "string"
            },
            "Model":{
              "type": "string",
              "description" : "simulatePlug model"
            }
          },
          "required": [ "InstanceName", "Manufacturer","Model" ]
        }
    }
}
)";

static const std::string kInitSchema = R"(
{
    "$schema": "http://json-schema.org/draft-07/schema",
    "type": "object",
    "properties": {
        "Manufacturer": {
            "title": "Manufacturer",
            "type": "string",
            "default": "Unknown",
            "readOnly": false,
            "examples": [
                "FooDisplays"
            ],
            "pattern": "^.*$"
        },
		"Model": {
            "title": "Model",
            "type": "string",
            "default": "001",
            "readOnly": false
        }
		
    },
    "additionalProperties": false,
    "required": [
        "Manufacturer"
    ]
}
)";

static const std::string kResultSchema = R"(
{
  "$schema": "http://json-schema.org/draft-07/schema",
  "type": "object",
  "properties": {
    "CommandName": {
      "$id": "#/ipf/battery/commandName",
      "type": "object",
      "properties": {
        "Success": {
          "type": "boolean"
        },
        "Error": {
          "type": "string"
        }
      },
      "required": [ "Success", "Error" ]
    },
	"BatteryAttach": {
      "$id": "#/root/battery/BatteryAttach",
      "type": "object",
      "properties": {
        "InstanceName": {
          "type": "string"
        }
      },
      "additionalProperties": false,
      "required": [ "InstanceName" ]
    },
    "BatteryDetach": {
      "$id": "#/root/battery/BatteryDetach",
      "type": "object",
      "properties": {
        "InstanceName": {
          "type": "string"
        }
      },
      "additionalProperties": false,
      "required": [ "InstanceName" ]
    }
  }
}
)";

namespace Ipf
{
auto mockedBatteryDriver = ::MockedBatteryDriver();

static const SchemaMap kSchemaMap = {
	{SchemaType::DATA, kDataSchema},
	{SchemaType::COMMAND, kCommandSchema},
	{SchemaType::RESULT, kResultSchema},
	{SchemaType::EVENT, kEventSchema},
	{SchemaType::INIT, kInitSchema}};

SampleBatteryProvider::SampleBatteryProvider(IFramework* pfw, const std::string& config)
	: ProviderDelegate(pfw, kSchemaMap, config)
{
	// Initialize data
	namespace_["State"] = "charging";
	namespace_["ChildCell"]["State"] = "charged";
	namespace_["IsIonBattery"] = false;

	// Manufacturer and model are not a required parameter in the schema so we still
	// have to check that it's there in init_
	if (!config_[JsonFields::INIT].empty()) {

		if (config_[JsonFields::INIT].contains("Manufacturer")) {
			namespace_["Manufacturer"] = config_[JsonFields::INIT]["Manufacturer"];
		}
		if (config_[JsonFields::INIT].contains("Model")) {
			namespace_["Model"] = config_[JsonFields::INIT]["Model"];
		}
	}
	else {
		// Properties can be optionally passed in at init time
		// Define "manufacturer" and "model" property that defaults to BatteryManufacturer and 001
		// respectively but can be overriden
		namespace_["Manufacturer"] = "BatteryManufacturer";
		namespace_["Model"] = "001";
	}
}

SampleBatteryProvider::~SampleBatteryProvider()
{}

// UpdateNamespace is invoked every time a client calls GetNode().
void SampleBatteryProvider::UpdateNamespace(const string& path)
{
	namespace_["State"] = mockedBatteryDriver.getState();
}

void SampleBatteryProvider::SetValue(const string& path, const string& value)
{
	if (path == "State") {
		mockedBatteryDriver.setState(value);
	}
}

void SampleBatteryProvider::RegisterEvent(const std::string& path)
{
	std::cout << "Battery provider got an event registration for path " << path << std::endl;
}

void SampleBatteryProvider::UnregisterEvent(const std::string& path)
{
	std::cout << "Battery provider got an event unregistration for path " << path << std::endl;
}

// Provider's ExecuteCommand() is called every time a client calls
// ExecuteCommand() Client API.
std::string SampleBatteryProvider::ExecuteCommand(const string& path, const std::string& command)
{
	Json json_command = Json::parse(command);
	Json result;
	bool err = false;
	std::string errmsg = "OK";

	if (json_command.contains("CommandName")) {
		mockedBatteryDriver.takeAction(result);
	}

	if (mockedBatteryDriver.getBatteryPercentage() > 89) {
		// Events are asynchronous, but for demo, it is attached to the
		// ExecuteCommand call. Client applications can register for events with a
		// callback function. If registered, framework will execute client
		// application(s)'s callback function(s)
		Json event;
		event["BatteryFullyCharged"]["State"] = "charged";
		framework_->RaiseEvent(event.dump());
	}

	// Framework is OK and command is valid
	if (json_command.contains("SimulatePlug")) {
		// Create a configuration for the child display
		Json config;
		config[JsonFields::INIT]["Manufacturer"] = json_command["SimulatePlug"]["Manufacturer"];
		config[JsonFields::INIT]["Model"] = json_command["SimulatePlug"]["Model"];

		std::string instance_name = json_command["SimulatePlug"]["InstanceName"];
		framework_->CreateChildInstance("Company.Org.Battery", instance_name, config.dump());

		result["SimulatePlug"] = {{"Success", !err}, {"Error", errmsg}};
		namespace_["Instances"][instance_name] = {};

		// Fire an event to notify any listeners that a new battery is attached
		Json event;
		event["BatteryAttach"]["InstanceName"] = instance_name;
		framework_->RaiseEvent(event.dump());
	}

	else if (json_command.contains("SimulateUnplug")) {
		std::string instance_name = json_command["SimulateUnplug"]["InstanceName"];
		framework_->DeleteChildInstance(instance_name);

		result["SimulateUnplug"] = {{"Success", !err}, {"Error", errmsg}};
		namespace_["Instances"].erase(instance_name);

		// Fire an event to notify any listeners that a battery was detached
		Json event;
		event["BatteryDetach"]["InstanceName"] = instance_name;
		framework_->RaiseEvent(event.dump());
	}

	// To be extended to include other advanced features. TBD

	return result.dump();
}
} // namespace Ipf
